/*
 *****    homebrew-radios.net   *****

 *****     Project TxrAVR       *****

 taRTC.c    program file
 
 Real time clock (I2C) bus driver program file

*/


#include <stdint.h>
#include "taRTC.h"
#include <compat/twi.h>
#include <string.h>
#include <avr/pgmspace.h>
#include "taTWI.h"
#include "taGlobal.h"
#include "taFT245R.h"
#include "taEATFTmenus.h"
#include <util/delay.h>
#include "taIntEeprom.h"
#include "taEEPROM.h"
#include "taKS0108Menus.h"
#include "taGraphicsEA320.h"

#define EatftRtcLeft 280;
#define EatftRtcTop 250;

// five RTCs supported
// re software and addressing:  
//  DS3232 identical to DS3231
//  PCF8562 identical to Epson RTC-8564

#define ISL1220 1
#define PCF8563 2   // and RTC-8564
#define DS3231 3   // and DS3232

#define DS3231_ADDRESS 0x68
#define ISL1220_ADDRESS 0x6F
#define PCF8563_ADDRESS 0x51


// Table of days in months
static const uint8_t MonthDays[12] PROGMEM  
     = { 31,28,31,30,31,30,31,31,30,31,30,31 };

// Table number of days from 1st Jan to start of month		 
static const uint16_t BeforeMonthDays[13] PROGMEM
     = {0,31,59,90,120,151,181,212,243,273,304,334,365};	// need 13 as next month after DEC	   

static const char MonthNames[12][4] PROGMEM
     = { "Jan","Feb","Mar","Apr","May","Jun","Jul","Aug","Sep","Oct","Nov","Dec" };

uint8_t sec;
uint8_t RtcActive;
uint8_t RtcFlag;
uint8_t RtcStatus;
uint8_t RtcKind;
uint16_t TestSeconds;
uint8_t LocalTimeOffset;
uint8_t disable = 1;               //Ray; 0 to disable it

void rtcDrawDateTime();
void rtcDrawSeconds();

uint8_t rtcBcdToBin(uint8_t bcd);
uint8_t rtcBinToBcd(uint8_t bin);
uint8_t rtcMonthDays(uint8_t m, uint8_t y);

void rtcBuildTime(struct Rtc_type* pRtc, char* StrTime );
void rtcBuildDate(struct Rtc_type* pRtc, char* StrDate, uint8_t ks);
void LeadingZero(uint8_t n, char* pS);
void rtcDeltaMinutes(struct Rtc_type* pRtcU, struct Rtc_type* pRtcL, uint8_t offs);
void rtcReadRtc(struct Rtc_type*  pRtc);
void rtcSetRtc(struct Rtc_type* pRtc);


////////////////////////////////////////////////////////////////////////////////////

void rtcInit(uint8_t DoCheck) __attribute__((section(".highmem")));
void rtcInit(uint8_t DoCheck)
{
	uint8_t b;
	NoClockDraw = 0;
	RtcActive = 0;
	RtcStatus = HardwareSettings.hsRtcStatus;
	RtcKind = RtcStatus & 0x0F;
	if (RtcStatus == 0) return;

	//////////////////////////////
	// RTCpos: used only by 320x240 mono graphics
	RTCpos = 1;   //  locate RTC at bottom of screen when no touch pad or bottom buttons
	{
		if (HaveTP || HaveBB)
		{
		  RTCpos = 2;   // locate RTC in help window area if DDS76 = 0 or 1 (ie DSP help switched off)
  	  if (DDSparams.dpDSPDDSKeysDisplay >= 2) RTCpos = 3;  // otherwise locate RTC in FreqInfo window
    }
	}  
	//////////////////////////
	switch (RtcKind)
	{
	  case (ISL1220):
		{
			RTC_I2C_ADDRESS = ISL1220_ADDRESS;
			break;
		}
	  case (PCF8563): 
		{
		  RTC_I2C_ADDRESS = PCF8563_ADDRESS;
		  rtcWriteByte(0,0);  // status regsiters must both be zero 
			rtcWriteByte(1,0);	
	    break;
    }
		case (DS3231):
		{
		  RTC_I2C_ADDRESS = DS3231_ADDRESS;
			rtcReadByte(0x0E,&b);   // read status register
      b &= 0x18;   //  all status bits except 3 and 4 set to zero  (no BBSQW)
			rtcWriteByte(0x0E,b);    // set status register
			break;
    }
  }

	sec = 0;
	ie_eeprom_read_block(&LocalTimeOffset, &ee.EE_LocalTimeOffset, 1);

// test to see if seconds moves - ie clock running
// we need a delay - so we use the DSP load time
  RtcActive = 1;
	rtcReadRtc(&RtcU);
	TestSeconds = RtcU.SC + 60*RtcU.MN;
  if(DoCheck) RtcActive = 0;
}

// calculates Local data/time (pRtcL-> )  from UTC  (pRtc->U)
void rtcDeltaMinutes(struct Rtc_type* pRtcU, struct Rtc_type* pRtcL, uint8_t offs) __attribute__((section(".highmem")));
void rtcDeltaMinutes(struct Rtc_type* pRtcU, struct Rtc_type* pRtcL, uint8_t offs)
{
	uint8_t F29, m;
	uint16_t md;
	uint32_t n,r,yr,mo,dy,hr,mn;
	uint32_t bmd = pgm_read_word(&BeforeMonthDays[pRtcU->MO - 1]);
	yr = pRtcU->YR;
	mo = pRtcU->MO;
	dy = pRtcU->DY;
	hr = pRtcU->HR;
	mn = pRtcU->MN;

	if((mo > 2)&&((yr % 4)==0)){bmd += 1;};
	n = mn + 60*(hr + 24*((bmd + (dy-1)) + 365*yr  + ((3 + yr) / 4)));
	if((offs & 0x80)==0)
	{n += 30*offs;}
	else
	{
	  offs ^= 0xFF;
		n -= 30*(offs + 1);
  }
  pRtcL->MN = n%60;
	n /= 60;   // n now = hours
	pRtcL->HR = n%24;
	n /= 24;   // n now = days
	pRtcL->YR = 4*(n/1461);
	r = n%1461;
	if(r>=366) //if not 2008,2012 etc then 2009,2010 or 2011 etc 
	{
	  pRtcL->YR += 1; 
		r -= 366;   // remove the leap year days
		F29 = 0;    // no 29th Feb 
	  pRtcL->YR += r / 365;   //add the 0,1, or 2 years after the leap year
		r %= 365;
	}
  else 
	{F29 = 1;}
	r += 1;   // if r = 0 then 1st Jan
	for (m=1;m<=12;m++)
	{
	  md = pgm_read_word(&BeforeMonthDays[m]);  // this is the next month (hence 1-13  = Feb to next Jan) 
		if(m>=2) md += F29;   // if March or later then add leap year Feb
		if(r<=md)
		{
			md = pgm_read_word(&BeforeMonthDays[m-1]);
			if(m>=3) md += F29;   // if m-1 is March or later then add leap year Feb

			pRtcL->DY = (r - md); break;}
	}   
  pRtcL->MO = m; 
}
	 
	 



// test if clock running - if not try to start it - then test again 
// only reason for doing this is for a clock that needs a start up action
// eg: ISL1220 - spefied sequencr.   PCF8563 - none. DS... need to load a dummy time initially
void rtcClockCheck() __attribute__((section(".highmem")));
void rtcClockCheck()
{
	if (RtcStatus == 0) return;
	RtcActive = 1;
	uint8_t b;
	rtcReadRtc(&RtcU);
	switch (RtcKind)
	{
	  case (ISL1220):
		{
			if(TestSeconds == (RtcU.SC + 60*RtcU.MN))
			{
    		// seconds not changed during DSP load - so start clock
	  		rtcReadByte(7,&b);
				b |= 0x10;
				rtcWriteByte(7,b);
				rtcWriteByte(0,0);  // write to the seconds counter to start the clock
				b &= ~(0x10);
				rtcWriteByte(7,b);
				// don't bother to  recheck - user can see if clock running	
			}
	  }
		case (PCF8563): break;
		case (DS3231): break;  // no action - osc autostarts first time Vcc is applied

	}
	rtcDoDateTime();
}

void rtcDoDateTime() __attribute__((section(".highmem")));
void rtcDoDateTime()
{
	rtcReadRtc(&RtcU);	
  rtcDeltaMinutes(&RtcU,&RtcL,LocalTimeOffset);
	rtcDrawDateTime();
}


uint8_t rtcMonthDays(uint8_t m, uint8_t y) __attribute__((section(".highmem")));
uint8_t rtcMonthDays(uint8_t m, uint8_t y)
{
  uint8_t d = pgm_read_byte(MonthDays[m]);
	if((m==2)&&((y%4)==0)){d += 1;};
	return d;
}


   
uint8_t rtcBinToBcd(uint8_t bin) __attribute__((section(".highmem")));
uint8_t rtcBinToBcd(uint8_t bin)
{
  return (bin % 10) + 16*(bin/10);  
}


uint8_t rtcBcdToBin(uint8_t bcd) __attribute__((section(".highmem")));
uint8_t rtcBcdToBin(uint8_t bcd)
{
  return (bcd % 16) + 10*(bcd/16);  
}



void rtcUsbRequest(char cmd) __attribute__((section(".highmem")));
void rtcUsbRequest(char cmd)
{
	uint8_t b,r,d, tries;
	uint8_t DoUSB;
	switch(cmd)
	{
		case 'A': {					// set Local time offset 8 bit 2s complement as units of 30 minutes
								if (usbReceiveBlock((uint8_t*)(&LocalTimeOffset), 1, 500) == 1)
								{ 
									ie_eeprom_write_block(&LocalTimeOffset, &ee.EE_LocalTimeOffset, 1);
									rtcDoDateTime();
								}
		            break;
							}							
	  case 'B': {         // read Local tine offset to PC
								ie_eeprom_read_block(&LocalTimeOffset, &ee.EE_LocalTimeOffset, 1);
								usbSendBlock((uint8_t*)(&LocalTimeOffset),1,500);
		            break;
							}
		case 'C': {					// set ATR (analogue trimming register) ISL1220 only
								if(RtcKind != ISL1220) return;
								if (usbReceiveBlock((uint8_t*)(&b), 1, 500) == 1)
								{ 
									b &= 0x3F;
									rtcReadByte(0x0A,&r);
									r &= 0xC0;      // preserved BMATR 0 and 1
									b |= r;
									rtcWriteByte(0x0A,b);
								}
		            break;
							}
							
	  case 'D': {         // read ATR (analogue trimming register)  ISL1220 only
								if(RtcKind != ISL1220) return;
								rtcReadByte(0x0A,&b);
								b &= 0x3F;
								usbSendBlock((uint8_t*)(&b),1,500);
		            break;
							}	 												 
		case 'E': {					// set Aging register trimming) DS3231/DS3232 only
								if(RtcKind != DS3231) return;
								if (usbReceiveBlock((uint8_t*)(&b), 1, 500) == 1)
								{ 
									rtcWriteByte(0x10,b);   // write to aging register
									// need to force temp conversion for immediate effcet of aging register change
									tries = 0;
									do
									{ 
										_delay_ms(1);
									  rtcReadByte(0x0F,&b);
									  tries += 1;
									}
									while (((b & 0x04)!=0) && (tries < 20)); // wait for BSY = 0
									rtcReadByte(0x0E,&b);
									b |= 0x20;  // set CONV bit
									rtcWriteByte(0x0E,b);
								}
		            break;
							}
							
	  case 'F': {         // read Aging register (trimming)  DS3231 / DS3232 only
								if(RtcKind != DS3231) return;
								rtcReadByte(0x10,&b);
								usbSendBlock((uint8_t*)(&b),1,500);
		            break;
							}	 												 
	  case 'R': {   // hardware options  - in taGlobal.h - read back to PC request
								rtcReadRtc(&RtcU);
								usbSendBlock((uint8_t*)(&RtcU),7,500);
								break;
              } 
	  case 'S': {   // hardware options  - in taGlobal.h - write from PC request
								if (usbReceiveBlock((uint8_t*)(&RtcU), 7, 500) == 1)
								{ 
									rtcSetRtc(&RtcU);
								  rtcDeltaMinutes(&RtcU,&RtcL,LocalTimeOffset);
									rtcDrawDateTime();
								}
								break;
              }
		case 'X': {					// set Fout
								if (usbReceiveBlock((uint8_t*)(&b), 1, 500) == 1)
								{ 
									if (RtcKind == ISL1220)
									{
										b &= 0x0F;
										b |= 0x10;      // No Fout on battery backup... Alarm INT and LPmode disabled
										rtcWriteByte(0x08,b);
									}
									if (RtcKind == PCF8563)
									{
										if (b > 0)
										{
										  b -= 1;   // drop down list index 1  - 4   becomes 0 - 3
											b |= 0x80;  //   set FE bit
                    }
										rtcWriteByte(0x0D,b);   
									}
									if (RtcKind == DS3231)
									{
									  b &= 0x03;
										rtcReadByte(0x0E,&d);
                    d &= 0xE7;   // clear bits 3 and 4
										d |= (b << 3);
										rtcWriteByte(0x0E,d);   
									}
								}
		            break;
							}
							
	  case 'Y': {         // read Fout
								DoUSB = 0;
								if (RtcKind == ISL1220)
								{
								  rtcReadByte(8,&b);
									b &= 0x0F;
									DoUSB = 1;
								}
								if (RtcKind == PCF8563)
								{
								  rtcReadByte(0x0D,&b);
									if ((b & 0x80) >  0)
									{
									  b &= 0x03;  // FD[1:0]
										b += 1;     // convert 0-3 to index 1 - 4
			            }
									else b = 0;
									DoUSB = 1;
								}
								if (RtcKind == DS3231)
								{
								  rtcReadByte(0x0E,&b);
									b &= 0x18;   // bits 3 an 4 only
									b = (b >> 3);
									DoUSB = 1;
								}
								if (DoUSB) usbSendBlock((uint8_t*)(&b),1,500);
                break;
							}	 												 
  }
}


void rtcPrepareTime() __attribute__((section(".highmem")));
void rtcPrepareTime()
{
	strcpy(StrA,PMS(se_UTC____));
	strcat(StrA,StrTimeU);
	strcat(StrA," ");
	strcpy(StrB,PMS(se_Local__));
	strcat(StrB,StrTimeL);
	strcat(StrB," ");
}



void rtcDrawDateTime() __attribute__((section(".highmem")));
void rtcDrawDateTime()
{
	if (zt || zg)
	{ 
		rtcBuildTime(&RtcU, StrTimeU);
		rtcBuildDate(&RtcU, StrDateU,0);
	}		
  rtcBuildTime(&RtcL, StrTimeL);
	rtcBuildDate(&RtcL, StrDateL,zs);
	if(zt) emeDrawDateTime();
	if(zs) ksmDrawDateTime();
	if(zg) geDrawDateTime();
}



void rtcDrawTime() __attribute__((section(".highmem")));
void rtcDrawTime()
{
	rtcBuildTime(&RtcU,StrTimeU);
  rtcBuildTime(&RtcL,StrTimeL);
	if(zt) emeDrawTime();
	if(zs) ksmDrawTime();
	if(zg) geDrawTime();
}


void LeadingZero(uint8_t n, char* s) __attribute__((section(".highmem")));
void LeadingZero(uint8_t n, char* s)
{
 	s[0] = 0;
	if(n<10) strcpy(s,"0");
	char  ss[3];
	itoa(n,ss,10);
	strcat(s,ss);
}


void rtcBuildTime(struct Rtc_type* pRtc, char* StrTime) __attribute__((section(".highmem")));
void rtcBuildTime(struct Rtc_type* pRtc, char* StrTime)
{
	LeadingZero(pRtc->HR, StrB);   // **************************   temporary
	strcpy(StrTime,StrB);	
	strcat(StrTime, ":");
	LeadingZero(pRtc->MN, StrB);
	strcat(StrTime, StrB);
	strcat(StrTime, ":");
	LeadingZero(pRtc->SC, StrB);
	strcat(StrTime, StrB);
	strcat(StrTime,"  ");
}


void rtcBuildDate(struct Rtc_type* pRtc, char* StrDate, uint8_t ks) __attribute__((section(".highmem")));
void rtcBuildDate(struct Rtc_type* pRtc, char* StrDate, uint8_t ks)
{
	uint8_t m;
	uint16_t y;
	LeadingZero(pRtc->DY, StrB);
	strcpy(StrDate,StrB);	
	strcat(StrDate, " ");
	m = pRtc->MO;
	if((m>0)&&(m<13)) strcat(StrDate,PMS(MonthNames[m-1])); else strcat(StrDate,"XXX");
	strcat(StrDate, " ");
	y = pRtc->YR;
	if(ks==0) y += 2000; 
	itoa(y, StrB, 10);
//	StrB[4] = 0;
	strcat(StrDate, StrB);
}




void rtcDoRTC() __attribute__((section(".highmem")));
void rtcDoRTC()
{
	if(!RtcFlag) return;
	RtcFlag = 0;
	if(!RtcActive) return;
	uint8_t b,s;
	switch (RtcKind)
	{
	  case ISL1220: rtcReadByte(0,&b);  break; // seconds
		case PCF8563: rtcReadByte(2,&b); break;  // seconds
		case DS3231:  rtcReadByte(0,&b); break;  // seconds
  }
	s = rtcBcdToBin(b);
  if (s == sec){return;}
	else
	{
		sec = s;
		if(sec==0)
		{
			rtcReadRtc(&RtcU);
      rtcDeltaMinutes(&RtcU,&RtcL,LocalTimeOffset);
			rtcDrawDateTime();
		}
  	else
		{
			RtcU.SC = sec;
			RtcL.SC = sec;
			rtcDrawTime();
		}
	}
}



void rtcReadRtc(struct Rtc_type*  pRtc) __attribute__((section(".highmem")));
void rtcReadRtc(struct Rtc_type*  pRtc)
{
	switch (RtcKind)
	{
    case ISL1220:
		{			
			rtcReadByte(0,&rtcbuf[0]);
		  rtcReadByte(1,&rtcbuf[1]);
		  rtcReadByte(2,&rtcbuf[2]); 
		  rtcReadByte(3,&rtcbuf[3]);
		  rtcReadByte(4,&rtcbuf[4]);
		  rtcReadByte(5,&rtcbuf[5]);
		  rtcReadByte(6,&rtcbuf[6]);
			pRtc->SC = rtcBcdToBin(rtcbuf[0]);
			pRtc->MN = rtcBcdToBin(rtcbuf[1]);
			pRtc->HR = rtcBcdToBin(rtcbuf[2] & 0x7F);  // mask 24 clock bit (MIL)
			pRtc->DY = rtcBcdToBin(rtcbuf[3]);
			pRtc->MO = rtcBcdToBin(rtcbuf[4]);
			pRtc->YR = rtcBcdToBin(rtcbuf[5]);
			pRtc->DW = rtcBcdToBin(rtcbuf[6]);
			break;
    }
		case PCF8563:
		{			
			rtcReadByte(2,&rtcbuf[0]);    // sec
		  rtcReadByte(3,&rtcbuf[1]);    // min
		  rtcReadByte(4,&rtcbuf[2]);    // hour
		  rtcReadByte(5,&rtcbuf[3]);    // day
		  rtcReadByte(6,&rtcbuf[4]);    // day of week - not used
		  rtcReadByte(7,&rtcbuf[5]);    // century/month
		  rtcReadByte(8,&rtcbuf[6]);    // years
			pRtc->SC = rtcBcdToBin(rtcbuf[0] & 0x7F);
			pRtc->MN = rtcBcdToBin(rtcbuf[1] & 0x7F);
			pRtc->HR = rtcBcdToBin(rtcbuf[2] & 0x3F); 
			pRtc->DY = rtcBcdToBin(rtcbuf[3] & 0x3F) ;
			pRtc->MO = rtcBcdToBin(rtcbuf[5] & 0x1F);
			pRtc->YR = rtcBcdToBin(rtcbuf[6]) + 100*(rtcbuf[5] >> 7); // rtcBUf[5] is century/month    century in bit 7
			break;
    }
		case DS3231:
		{			
			rtcReadByte(0,&rtcbuf[0]);    // sec
		  rtcReadByte(1,&rtcbuf[1]);    // min
		  rtcReadByte(2,&rtcbuf[2]);    // hour
		  rtcReadByte(3,&rtcbuf[3]);    // day of week  - unused
		  rtcReadByte(4,&rtcbuf[4]);    // day
		  rtcReadByte(5,&rtcbuf[5]);    // century/month
		  rtcReadByte(6,&rtcbuf[6]);    // years
			pRtc->SC = rtcBcdToBin(rtcbuf[0] & 0x7F);
			pRtc->MN = rtcBcdToBin(rtcbuf[1] & 0x7F);
			pRtc->HR = rtcBcdToBin(rtcbuf[2] & 0x3F); 
			pRtc->DW = rtcBcdToBin(rtcbuf[3] & 0x07);
			pRtc->DY = rtcBcdToBin(rtcbuf[4] & 0x3F);
			pRtc->MO = rtcBcdToBin(rtcbuf[5] & 0x1F);
			pRtc->YR = rtcBcdToBin(rtcbuf[6]) + 100*(rtcbuf[5] >> 7); // rtcBUf[5] is century/month    century in bit 7
			break;
    }
  }
}




void rtcSetRtc(struct Rtc_type* pRtc) __attribute__((section(".highmem")));
void rtcSetRtc(struct Rtc_type* pRtc)
{
  uint8_t b;
	switch (RtcKind)
	{
    case ISL1220:
		{
	  	rtcReadByte(7,&b);
			b |= 0x10;
			rtcWriteByte(7,b);
  		rtcbuf[0] = rtcBinToBcd(pRtc->SC);
  		rtcbuf[1] = rtcBinToBcd(pRtc->MN);
  		rtcbuf[2] = rtcBinToBcd(pRtc->HR) + 0x80;   // set bit 7 for 24 hour clock
  		rtcbuf[3] = rtcBinToBcd(pRtc->DY);
  		rtcbuf[4] = rtcBinToBcd(pRtc->MO);
  		rtcbuf[5] = rtcBinToBcd(pRtc->YR);	
			rtcBlockWrite(0,6);
			b &= ~(0x10);
			rtcWriteByte(7,b);
			break;
  	}	
    case PCF8563:
		{
  		rtcbuf[0] = rtcBinToBcd(pRtc->SC);
  		rtcbuf[1] = rtcBinToBcd(pRtc->MN);
  		rtcbuf[2] = rtcBinToBcd(pRtc->HR) & 0x3F;
  		rtcbuf[3] = rtcBinToBcd(pRtc->DY);                              
  		rtcbuf[4] = 0;                          // day of week - not used
  		rtcbuf[5] = rtcBinToBcd(pRtc->MO) ;
			if (pRtc->YR >100) rtcbuf[5] |= 0x80;
  		rtcbuf[6] = rtcBinToBcd(pRtc->YR %100);	
//			rtcBlockWrite(2,7);         //  this should work as an alternative
		  rtcWriteByte(2,rtcbuf[0]); 
		  rtcWriteByte(3,rtcbuf[1]); 
		  rtcWriteByte(4,rtcbuf[2]); 
		  rtcWriteByte(5,rtcbuf[3]); 
		  rtcWriteByte(6,rtcbuf[4]); 
		  rtcWriteByte(7,rtcbuf[5]); 
		  rtcWriteByte(8,rtcbuf[6]); 
			break;
  	}	
    case DS3231:
		{
  		rtcbuf[0] = rtcBinToBcd(pRtc->SC);
  		rtcbuf[1] = rtcBinToBcd(pRtc->MN);
  		rtcbuf[2] = rtcBinToBcd(pRtc->HR) &0x3F;   // clear bit 6 for 24 hour clock
  		rtcbuf[3] = 0;                             // day of week - unused;
			rtcbuf[4] = rtcBinToBcd(pRtc->DY) & 0x3F;
  		rtcbuf[5] = rtcBinToBcd(pRtc->MO);
			if (pRtc->YR >100) rtcbuf[5] |= 0x80;  // century bit
  		rtcbuf[6] = rtcBinToBcd(pRtc->YR % 100);	
//			rtcBlockWrite(0,7);
		  rtcWriteByte(0,rtcbuf[0]); 
		  rtcWriteByte(1,rtcbuf[1]); 
		  rtcWriteByte(2,rtcbuf[2]); 
		  rtcWriteByte(3,rtcbuf[3]); 
		  rtcWriteByte(4,rtcbuf[4]); 
		  rtcWriteByte(5,rtcbuf[5]); 
		  rtcWriteByte(6,rtcbuf[6]); 
			break;
  	}	
	}
}








////////////////////////////////////////////////////














