/*
 *****    homebrew-radios.net   *****

 *****     Project TxrAVR       *****


 taProgmen.c    program file
 

Handles constant date stored in program (flash) memory
This includes literal strings which are here rather than in-line

Reason: ATmega2560 code cannot directly access flash memory

Example: 
         void DisplayString("Have a nice day");
This results in the string being copied to RAM before the
Main() routine begins.   This potentially wastes valuable RAM

Note also that pointers are 16 bit so problems accessing beyond 64k in flash.   

Solution:  use PROGMEM to place strings in flash and return a pointer to 
the string's flash location (and not its pre-run-time RAM location).
Either use FAR calls to flash or make sure all PROGMEM data is 
located below 64k

Data labels prefixed pmd  (ProgMemData)


// ************ IMPORTANT ********************
//  Strings are returned in:   char pmStrBuf[80];   defined in this module
//  DO NOT MAKE A SECOND CALL TO THE ROUTINE UNTIL YOU HAVE OFF-LOADED THE THE FIRST STRING
//  (otherwise the second will overwrite the first in pmStrBuf[80] )
// ********************************************
*/

#include "taGlobal.h"
#include "taStarDSP.h"
#include "taDebug.h"
#include "taStarControl.h"
#include "ta24LC512.h"

// Band port bit patterns - for output to diode matrix
#define BandBit18   0x01
#define BandBit35   0x02
#define BandBit5    0x04
#define BandBit7    0x08
#define BandBit10   0x10
#define BandBit14   0x20
#define BandBit1821 0x40
#define BandBit2428 0x80


// Band port bit pattern list
static const uint8_t pmdBandBitList[10] PROGMEM = {
		BandBit18,
		BandBit35,
		BandBit5,
		BandBit7,
		BandBit10,
		BandBit14,
		BandBit1821,
		BandBit1821,
		BandBit2428,
		BandBit2428
};

// Band codes for DSP unit
static const uint8_t pmdDSPbands[10] PROGMEM = {10,9,5,8,7,6,4,3,2,1};

// Table of userno sequence
static const uint8_t pmdSeqToUserno[48] PROGMEM = {
	11, 12, 13, 14, 21, 22, 23, 24, 25, 31,		//  0 -  9
    32, 33, 41, 42, 43, 44, 45, 46, 47, 51,		// 10 - 19
	52, 53, 54, 55, 56, 57, 58, 61, 62, 63,		// 20 - 29
	64, 65, 66, 71, 72, 73, 74, 75, 76, 81,		// 30 - 39
	82, 83, 84, 85, 91, 92, 93, 99,				// 40 - 47  47 is "not assigned"
};

// Table to get seq position from userno
static const uint8_t pmdUsernoToSeq[100] PROGMEM = {
	255,255,255,255,255,255,255,255,255,255,	//  0 -  9
    255,  0,  1,  2,  3,255,255,255,255,255,	// 10 - 19
	255,  4,  5,  6,  7,  8,255,255,255,255,	// 20 - 29
	255,  9, 10, 11,255,255,255,255,255,255,	// 30 - 39
	255, 12, 13, 14, 15, 16, 17, 18,255,255,	// 40 - 49
	255, 19, 20, 21, 22, 23, 24, 25, 26,255,	// 50 - 59
	255, 27, 28, 29, 30, 31, 32,255,255,255,	// 60 - 69
	255, 33, 34, 35, 36, 37, 38,255,255,255,	// 70 - 79
	255, 39, 40, 41, 42, 43,255,255,255,255,	// 80 - 89
	255, 44, 45, 46,255,255,255,255,255, 47,	// 90 - 99
};

// Table of userno sequence with x params
static const uint8_t pmdSeqToXUserno[60] PROGMEM = {
	11, 12, 13, 14, 15, 16, 17, 18, 19, 21,		//  0 -  9
    22, 23, 24, 25, 26, 27, 28, 29, 31, 32,		// 10 - 19
	33, 41, 42, 43, 44, 45, 46, 47, 51, 52,		// 20 - 29
	53, 54, 55, 56, 57, 58, 61, 62, 63, 64,		// 30 - 39
	65, 66, 71, 72, 73, 74, 75, 76, 77, 78,		// 40 - 49
	81, 82, 83, 84, 85, 86, 91, 92, 93, 99,		// 50 - 59  59 is "not assigned"
};

// Table to get seq position from userno
static const uint8_t pmdXUsernoToSeq[100] PROGMEM = {
	255,255,255,255,255,255,255,255,255,255,	//  0 -  9
    255,  0,  1,  2,  3,  4,  5,  6,  7,  8,	// 10 - 19
	255,  9, 10, 11, 12, 12, 14, 15, 16, 17,	// 20 - 29
	255, 18, 19, 20,255,255,255,255,255,255,	// 30 - 39
	255, 21, 22, 23, 24, 25, 26, 27,255,255,	// 40 - 49
	255, 28, 29, 30, 31, 32, 33, 34, 35,255,	// 50 - 59
	255, 36, 37, 38, 39, 40, 41,255,255,255,	// 60 - 69
	255, 42, 43, 44, 45, 46, 47, 48, 49,255,	// 70 - 79
	255, 50, 51, 52, 53, 54, 55,255,255,255,	// 80 - 89
	255, 56, 57, 58,255,255,255,255,255, 59,	// 90 - 99
};

// Table to get name list position from starno
static const uint8_t pmdParamLookupNL[100] PROGMEM = {
	255,255,255,255,255,255,255,255,255,255,	//  0 -  9
	255,  0,  1,  2,  3,  4,  5,  6,  7,  8,	// 10 - 19
	255,  9, 10, 11, 12, 13, 14, 15, 16, 17,	// 20 - 29
	255, 18, 19, 20,255,255,255,255,255,255,	// 30 - 39
	255, 21, 22, 23, 24, 25, 26, 27,255,255,	// 40 - 49
	255, 28, 29, 30, 31, 32, 33, 34, 35,255,	// 50 - 59
	255, 36, 37, 38, 39, 40, 41, 42, 43, 44,	// 60 - 69
	255, 45, 46, 47, 48, 49, 50, 51, 52, 53,	// 70 - 79
	255, 54, 55, 56, 57, 58, 59,255,255,255,	// 80 - 89
	255, 60, 61, 62, 63, 64,255,255,255, 65,	// 90 - 99
};


// band margins in kHz
// to read this data use:
//  f = 1000*pgm_read_word(&pmdLowFreqLimit[band]);    etc 
static const uint16_t pmdLowFreqLimit[]  PROGMEM = {1800,3500,5200,7000,10100,14000,18068,21000,24890,28000};
static const uint16_t pmdHighFreqLimit[] PROGMEM = {2000,3800,5400,7200,10150,14350,18168,21450,24990,29700};

static const char pmdThirtySpaces[] PROGMEM = "                              ";

char pmStrBuf[80];	// buffer used to return strings


//  Return string of spaces
char* pmdSpaces(uint8_t n)
{
	strcpy_P(pmStrBuf, (PGM_P)pmdThirtySpaces);
	pmStrBuf[n] = 0;
	return &pmStrBuf[0];
}


// GetBandPort value
uint8_t pmGetBandPort(uint8_t b)  
{
	if (WideBand == 1) return 0;
	else return pgm_read_byte(&pmdBandBitList[b]);
}


//  Get button caption from number
char* pmButtonCap(uint8_t n)
{
	strcpy_P(pmStrBuf, (PGM_P)&pmdButtonCapList[n]);
	return &pmStrBuf[0];
}  


//  Get mode name from number
char* pmModeName(uint8_t n)
{
	strcpy_P(pmStrBuf, (PGM_P)&pmdModeNameList[n]);
	return &pmStrBuf[0];
}  


//  Get band name from number
char* pmBandName(uint8_t n)
{
	strcpy_P(pmStrBuf, (PGM_P)&pmdBandNameList[n]);
	return &pmStrBuf[0];
}  


// Get parameter name from list index
char* pmParamNameFromListIndex(uint8_t i)
{
	strcpy_P(pmStrBuf, (PGM_P)&pmdParamNameList[i]);
	return &pmStrBuf[0];
}


// Get parameter name from list index
char* pmUnitsFromListIndex(uint8_t i)
{
	strcpy_P(pmStrBuf, (PGM_P)&pmdUnitsList[i]);
	return &pmStrBuf[0];
}


//  Get parameter name list index from parameter number
uint8_t pmParamNameIndexNL(uint8_t p)
{
	return pgm_read_byte(&pmdParamLookupNL[p]);
}	


//  Get parameter name from parameter number
char* pmParamNameNL(uint8_t p)
{
	uint8_t n;

	n = pgm_read_byte(&pmdParamLookupNL[p]);
	if (n == 255) pmStrBuf[0] = 0;   // empty string
	else strcpy_P(pmStrBuf, (PGM_P)&pmdParamNameList[n]);
	return &pmStrBuf[0];
}  


//  Get list index from parameter number
uint8_t  pmListIndexFromParamNo(uint8_t p)
{
	return pgm_read_byte(&pmdParamLookupNL[p]);
}


// Get userno from menu sequence index
uint8_t pmSeqToUserNo(uint8_t n)
{
	return pgm_read_byte(&pmdSeqToUserno[n]);
}


// Get menu sequence index from userno
uint8_t pmUsernoToSeq(uint8_t n)
{
	return pgm_read_byte(&pmdUsernoToSeq[n]);
}


int pmUserNoListShift(uint8_t olduserno, int shift, uint8_t AllowUnassigned)
{
	int i;
	uint8_t mx, userno;

	mx = 46;
	if (AllowUnassigned) mx = 47;
	i = -1;
	do {
		i++;
    	userno = pgm_read_byte(&pmdSeqToUserno[i]);
	} while((userno != olduserno) && (i < mx));
	i += shift;
	if (i < 0) i = (i%(mx+1)) ? ((i%(mx+1))+(mx+1)) : 0;
	else if (i > mx) i = i%(mx+1);
	return pgm_read_byte(&pmdSeqToUserno[i]);
}

 
// Get userno from menu sequence index with x params
uint8_t pmSeqToXUserNo(uint8_t n)
{
	return pgm_read_byte(&pmdSeqToXUserno[n]);
}


// Get menu sequence index from userno with x params
uint8_t pmXUsernoToSeq(uint8_t n)
{
	return pgm_read_byte(&pmdXUsernoToSeq[n]);
}


int pmXUserNoListShift(uint8_t olduserno, int shift, uint8_t AllowUnassigned)
{
	int i;
	uint8_t mx, userno;

	mx = 58;
	if (AllowUnassigned) mx = 59;
	i = -1;
	do {
		i++;
    	userno = pgm_read_byte(&pmdSeqToXUserno[i]);
	} while((userno != olduserno) && (i < mx));
	i += shift;
	if (i < 0) i = (i%(mx+1)) ? ((i%(mx+1))+(mx+1)) : 0;
	else if (i > mx) i = i%(mx+1);
	return pgm_read_byte(&pmdSeqToXUserno[i]);
}

 
//  Get parameter short name from parameter number
char* pmParamNameShortNL(uint8_t starno)
{
	uint8_t n;

	n = pgm_read_byte(&pmdParamLookupNL[starno]);
	if (n == 255) pmStrBuf[0] = 0;   // empty string
	else strcpy_P(pmStrBuf, (PGM_P)&pmdParamShortNameList[n]);
	return &pmStrBuf[0];
}  


char* pmParamNameShortLN(uint8_t n)
{
	if (n > 65) pmStrBuf[0] = 0; // empty string
	else strcpy_P(pmStrBuf, (PGM_P)&pmdParamShortNameList[n]);
	return &pmStrBuf[0];
}


char* PMS(PGM_P pfs)
{
	strcpy_P(pmStrBuf, pfs);
	return &pmStrBuf[0];
}


// Get band code for DSP
uint8_t pmGetDSPband(uint8_t b)
{
	return pgm_read_byte(&pmdDSPbands[b]);
}


//Get band low frequency limit
uint32_t pmGetBandLowLimit(uint8_t b)
{
	uint32_t lw;

	lw = pgm_read_word(&pmdLowFreqLimit[b]);
	return 1000*lw;
}


// Get band high frequency limit
uint32_t pmGetBandHighLimit(uint8_t b)
{
	uint32_t lw;

	lw = pgm_read_word(&pmdHighFreqLimit[b]);
	return 1000*lw;
}
