/*
 *****    homebrew-radios.net   *****

 *****     Project TxrAVR       *****


 taI2C.c    program file
 
I2C bus peripheral program file

*/

#include "taGlobal.h"
#include "taTWI.h"
#include "taStarControl.h"
#include "taEEPROM.h"
#include "taFT245R.h"
#include "taIntEeprom.h"
#include "taI2C.h"
#include "taDDS.h"
#include "taButtons.h"
#include "taGraphicsControl.h"
#include "taEATFT.h"

#define PCA9555_Input0  0
#define PCA9555_Input1  1
#define PCA9555_Output0 2
#define PCA9555_Output1 3
#define PCA9555_Invert0 4
#define PCA9555_Invert1 5
#define PCA9555_Config0 6
#define PCA9555_Config1 7


// PCA9555 band control words   {kind-1][Band]
// note index is kind-1     Kind = 0 = no IC2 BPF
static const uint16_t I2CBandBitList[2][10] PROGMEM = {
		{  // Picastar sequential. Note that Band is frequency derived
			0x0001,  // 160m
			0x0002,  // 80m
			0x0004,  // 60m
			0x0008,  // 40m
			0x0010,  // 30m
			0x0020,  // 20m
			0x0040,  // 17m
			0x0080,  // 15m
			0x0100,  // 12m
			0x0200   // 10m 
		},
		{  // PA3AKE   (NB  0x0200 is 6m. Upper 4 bits 0xFxxx are attenuators 2 4 8 16
			0x0001,  // 160m
			0x0002,  // 80m
			0x0400,  // 60m
			0x0004,  // 40m
			0x0008,  // 30m
			0x0010,  // 20m
			0x0020,  // 17m
			0x0040,  // 15m
			0x0080,  // 12m
			0x0100   // 10m 
		}
};

// CONTROL DATA  - copied in from EEPROM
// Button labels remain in EEPROM  only

static uint8_t BPF_Kind;
static uint8_t BPF_SlaveAddress;
static uint8_t PA3AKE_SquarerAddress;
static uint8_t PA3AKE_MixerAddress;
static uint8_t PA3AKE_MixerEnable;
static uint8_t PA3AKE_RooferSwitchEnable;
static uint16_t PA3AKE_MixerDAC;
static uint16_t PA3AKE_SquarerDAC;
static uint8_t PA3AKE_AttendB;
static char PA3AKE_Roofer;
static uint8_t EnablePCA9555;
static uint8_t grouper;
static uint8_t GroupStepTask[4];
static uint8_t BPF_Address;
static uint16_t TxRxDataword;
static uint8_t I2CPreviousTransmit;

char StepTaskLabel[4][12];
uint8_t UseStepTaskLabel[4];

void icSetDAC(char kind,uint16_t v);
uint16_t icPCA9555ReadWord(uint8_t Address);
void icPCA9555WriteBits(char ULB, uint8_t Address, uint16_t Data);
uint8_t icModifyRxTxDatawordBits(char SCT, uint16_t Dataword);
uint8_t icPCA9555ModifyBits(char SCT, uint8_t Address, uint16_t Dataword);
void icFindGroupMembers(uint8_t IndexTask, uint8_t IndexGroup, uint8_t Address);
void icSelectRoofer(char kind);
void icStepGroupsInit();
void icApplyRxTxDataword();


void icInit() 
{
	PA3AKE_AttendB = 0;
	uint16_t DacValue;
	uint8_t task, Group;
	uint8_t Opcode[8];
	uint16_t Dataword;
	uint8_t Address[8];
	ie_eeprom_read_block(&BPF_Kind, &ee.EE_I2C_BpfKind, 1);
	ie_eeprom_read_block(&BPF_SlaveAddress, &ee.EE_I2C_BpfAddress, 1);
	ie_eeprom_read_block(&PA3AKE_SquarerAddress, &ee.EE_PA3AKE_SquarerAddress, 1);
	ie_eeprom_read_block(&PA3AKE_MixerAddress, &ee.EE_PA3AKE_MixerAddress, 1);
	ie_eeprom_read_block(&PA3AKE_MixerEnable, &ee.EE_PA3AKE_MixerEnable, 1);
	ie_eeprom_read_block(&PA3AKE_RooferSwitchEnable, &ee.EE_PA3AKE_RooferSwitchEnable, 1);
	ie_eeprom_read_block(&EnablePCA9555, &ee.EE_EnablePCA9555, 1);
	
	///////   PA3AKE   Mixer init //////////////// 
	if(PA3AKE_MixerEnable)
	{
		ie_eeprom_read_block(&DacValue, &ee. EE_PA3AKE_MixerDAC, 2);
		icSetDAC('M',DacValue);
		ie_eeprom_read_block(&DacValue, &ee. EE_PA3AKE_SquarerDAC, 2);
		icSetDAC('S',DacValue);
	}

	//////////  PA3AKE roofer init ///////////////
	PA3AKE_Roofer = 'S'; // SSb
	icSelectRoofer('S');  // default to SSB on startup	

	//////////// BPF init /////////////
	if(BPF_Kind>0)
	{
	 	BPF_Address = 0x40 + (BPF_SlaveAddress << 1);
		I2Cbuf[0] = PCA9555_Output0;
		I2Cbuf[1] = 0;
		I2Cbuf[2] = 0;
		icBlockWrite(3,BPF_Address); // set all outputs to 0v
		I2Cbuf[0] = PCA9555_Config0;
		I2Cbuf[1] = 0x00;
		I2Cbuf[2] = 0x00;
		icBlockWrite(3,BPF_Address); // set all to output
	}
	////////////  PCA9555 task button system init
	if(EnablePCA9555>0)
	{
		for(Group=1;Group<=4;Group++) UseStepTaskLabel[Group-1] = 0;
			for(task=1;task<=8;task++)
		{		
			ie_eeprom_read_block(&Opcode[task-1], &ee.EE_I2C_ButtonOpcode[task-1],1);		
			if(Opcode>0)
			{
				ie_eeprom_read_block(&Address[task-1], &ee.EE_I2C_ButtonAddress[task-1],1);		
				icPCA9555WriteBits('B',Address[task-1],0);  // zero PCA9555 ... will occur more than once!
			}
		}	
		for(task=1;task<=8;task++)
		{				
			if(Opcode[task-1]==10)
			{
				ie_eeprom_read_block(&Dataword, &ee.EE_I2C_ButtonDataword[task-1],2);							
				icPCA9555WriteBits('B',Address[task-1],Dataword);
				TxRxDataword = Dataword;
			}
		}
		icStepGroupsInit();  // press one button of each stepper group to intialises caption and PCA955 output
	//	if(zg) gcGraphicsRedrawLabels(0,0);
	//	if(zt) etTouchpadCaptions();
	}
}


void icStepGroupsInit()
{
	uint8_t Opcode;
	uint8_t GroupDone[4] = {0,0,0,0};
	uint8_t Group;
	uint8_t I2Ctask;
	for(I2Ctask=1;I2Ctask<=8;I2Ctask++)
	{
		ie_eeprom_read_block(&Opcode, &ee.EE_I2C_ButtonOpcode[I2Ctask-1],1);		
		ie_eeprom_read_block(&Group, &ee.EE_I2C_ButtonGroup[I2Ctask-1],1);		
		if((Opcode>=7)&&(Opcode<=9)&&(GroupDone[Group]==0))
		{
			icPCA9555Task(I2Ctask);
			GroupDone[Group] = 1;
		}
	}
}




void icPA3AKE_ToggleRoofer()
{
	if(PA3AKE_Roofer=='C') PA3AKE_Roofer = 'S'; else PA3AKE_Roofer = 'C';
	icSelectRoofer(PA3AKE_Roofer);
}


void icSelectRoofer(char kind)   // PCF8674A  address = 0 1 1 1 A2 A1 A0
{
	uint8_t Address;
	if(PA3AKE_RooferSwitchEnable==0) return;
	ie_eeprom_read_block(&Address, &ee.EE_PA3AKE_RooferAddress, 1);
	if(PA3AKE_Roofer=='C') I2Cbuf[0] = 0x01; else I2Cbuf[0] = 0x00;	
	icBlockWrite(1,0x70 + (Address <<1));
}


void icSetDAC(char kind,uint16_t v)
{
	uint8_t Address;
	if(PA3AKE_MixerEnable==0) return;
	switch(kind)
	{
	  case 'M': Address = 0x18 + (PA3AKE_MixerAddress << 1); break;
		case 'S': Address = 0x18 + (PA3AKE_SquarerAddress << 1); break;
		default: return;
	}
	I2Cbuf[1] = v % 0x100;
	I2Cbuf[0] = (v / 0x100) & 0x0F;  // bits 6 adn 7 dont care. Bits 5 and 6 must be 0.
	icBlockWrite(2,Address); // set all to output
}



void icSetBPF()  // also sets attenuator
{
	uint16_t Mask,w;
	if(BPF_Kind==0) return;
	// PCA9555 has based address fixed at 0x40
	uint16_t ControlWord = pgm_read_word(&I2CBandBitList[BPF_Kind-1][Band]);  // Band from taDDS.h
	uint8_t AdB;
	if(Transmit) AdB = 0; else AdB = PA3AKE_AttendB;
	switch(BPF_Kind)
	{
		case 1: Mask = 0x03FF;    // picastar sequential 10 bits used. Leave other bits free
		case 2: 
		{
			ControlWord += ((AdB/2) << 12);  // add in the attenutor bits
			Mask = 0xFFFF;  // all 16 bits used 12 filter + ATTEN
		}
	}
	I2Cbuf[0] = PCA9555_Output0;
	icBlockWrite(1,BPF_Address); // write command byte
	icBlockRead(2,BPF_Address); // read dataword
	w = I2Cbuf[0] + 0x100*I2Cbuf[1]; //existing dataword in w
	w &= ~Mask;    // unmasked bits  (10-15 in case of picastar BPF)
	ControlWord &= Mask;
	w |= ControlWord;  //  controll bit and unmasked bit
	I2Cbuf[0] = PCA9555_Output0;
	I2Cbuf[1] = w % 0x100;
	I2Cbuf[2] = w / 0x100;
	icBlockWrite(3,BPF_Address);   // write back
}


void icI2CLoadRequest(char kind)
{
  uint8_t block[100];
	switch(kind)
	{
		case 'D': // Button task data
		{
			if(usbReceiveBlock((uint8_t*)(&block),40,500)==1)
			{ 
      	ie_eeprom_write_block(block, &ee.EE_I2C_ButtonOpcode[0], 40);
				icInit();
			}
			break;
    }
		case 'L': // Button task labels
		{
			if(usbReceiveBlock((uint8_t*)(&block),96,500)==1)
			{ 
      	ie_eeprom_write_block(block, &ee.EE_I2C_ButtonLabels[0][0], 96);
				icInit();
				}
			break;
    }
		case 'I': // other I2C data (BPF etc)
		{
			if(usbReceiveBlock((uint8_t*)(&block),24,500)==1)
			{ 
      	ie_eeprom_write_block(block, &ee.EE_I2C_BpfKind, 24);
				icInit();
			}
			break;
    }
		case 'M':  // mixer DAC adjust
		case 'S':  // squarer DAC adjust 
		{
			if(usbReceiveBlock((uint8_t*)(&block),2,500)==1)
			{ 
				uint16_t dac = block[0] + 0x100*block[1];
				icSetDAC(kind,dac);
				if (kind=='M')
				{
					PA3AKE_MixerDAC = dac;
					ie_eeprom_write_block(block, &ee.EE_PA3AKE_MixerDAC, 2);
				}	 
				else 
				{
					PA3AKE_SquarerDAC = dac;	
					ie_eeprom_write_block(block, &ee.EE_PA3AKE_SquarerDAC, 2);
				}
				icSetDAC(kind,dac);
			}
			break;
    }
	}
}

//  re DAC value writes to EEPROM
// during adjustment a write request is received about 1x / second 
// depsite the fact that the spin-Edit ocntrol on Hobcat is moving much faster
// Hobcat makes a rapid sequence of identical USB requests (  frmMain.USBstate[AD5321] := 'W';)
// but these are only services at about 1/ sec
// - thus - we wont wear out the EEPROM!!!


void icI2CSaveRequest(char kind)
{
  uint8_t block[100];
	switch(kind)
	{
		case 'D': 
		{
    	ie_eeprom_read_block(block, &ee.EE_I2C_ButtonOpcode[0], 40);
		  usbSendBlock((uint8_t*)(&block),40,500);
			break;
    }
		case 'L': 
		{
    	ie_eeprom_read_block(block, &ee.EE_I2C_ButtonLabels[0][0], 96);
		  usbSendBlock((uint8_t*)(&block),96,500);
			break;
    }
		case 'I': 
		{
    	ie_eeprom_read_block(block, &ee.EE_I2C_BpfKind, 24);
		  usbSendBlock((uint8_t*)(&block),24,500);
			break;
    }
		case 'C':  // mixer DAC value request
		{
    	ie_eeprom_read_block(block, &ee.EE_PA3AKE_MixerDAC, 4);
		  usbSendBlock((uint8_t*)(&block),4,500);
			break;
		}
	}
}




void icPCA9555WriteBits(char ULB, uint8_t Address, uint16_t Data)
{
	uint8_t n;
	I2Cbuf[0] = PCA9555_Config0;
	I2Cbuf[1] = 0x00;
	I2Cbuf[2] = 0x00;
	icBlockWrite(3,(0x40 + (Address << 1))); // set all to output
	if(ULB=='U')
	{ 
		I2Cbuf[0] = PCA9555_Output1;  // upper word write code
		I2Cbuf[1] = (Data >> 8);				// upper byte of Data
	}
	else
	{ 
		I2Cbuf[0] = PCA9555_Output0;
		I2Cbuf[1] = Data % 0x100;
		I2Cbuf[2] = Data / 0x100; // not used when writing lower byte only
	}
	if(ULB=='B') n=3; else n=2;  // 'B' = both bbytes n = 3
	icBlockWrite(n,(0x40 + (Address << 1))); // write dataword or upper or lower byte
}


uint16_t icPCA9555ReadWord(uint8_t Address)
{
	I2Cbuf[0] = PCA9555_Output0;
	icBlockWrite(1,(0x40 + (Address << 1))); // write command byte
	icBlockRead(2,(0x40 + (Address << 1))); // read dataword
	return I2Cbuf[0] + 0x100*I2Cbuf[1];
}



// return value (bits) only used for toggle
// Toggle function looks for all the specified bits to be 0 or all 1
// -  if not set them all to 1 ... then thereafter can toggle with EOR
// - return value used to highlight controlling button 
uint8_t icModifyRxTxDatawordBits(char SCT, uint16_t Dataword)
{
	uint8_t toggledon = 0;
	switch(SCT)
	{
	  case 'S': TxRxDataword |= Dataword; break;
	  case 'C': TxRxDataword &= ~Dataword; break;
	  case 'T': 
		{
			if( ( (TxRxDataword & Dataword) != 0) && (  (TxRxDataword &Dataword) != Dataword ) )
			{
				TxRxDataword |= Dataword;
				toggledon = 1;
			}
			else 
			{
				TxRxDataword ^= Dataword; 
				if((TxRxDataword &Dataword) == Dataword) toggledon = 1;
			}
			break;
		}
	}
	return toggledon;
}


// return value (bits) only used for toggle
// Toggle function looks for all the specified bits to be 0 or all 1
// -  if not set them all to 1 ... then thereafter can toggle with EOR
// - return value used to highlight controlling button 
uint8_t icPCA9555ModifyBits(char SCTW, uint8_t Address, uint16_t Dataword)
{
  uint16_t pcadata = icPCA9555ReadWord(Address);
	uint8_t toggledon = 0;
	switch(SCTW)
	{
	  case 'S': pcadata |= Dataword; break;
	  case 'C': pcadata &= ~Dataword; break;
	  case 'T': 
		{
			if( ( (pcadata & Dataword) != 0) && (  (pcadata &Dataword) != Dataword ) )
			{
				pcadata |= Dataword;
				toggledon = 1;
			}
			else 
			{
				pcadata ^= Dataword; 
				if((pcadata &Dataword) == Dataword) toggledon = 1;
			}
			break;
		}
		case 'W':
		{
			pcadata &= ~Dataword;
			pcadata |= (TxRxDataword & Dataword);
			break;
		}
	}
	icPCA9555WriteBits('B',Address,pcadata);  // 'B' = both bytes 
	return toggledon;
}


// for stepping groups, set index task = 0 so as to find whole group
void icFindGroupMembers(uint8_t IndexTask, uint8_t IndexGroup, uint8_t Address)
{
	uint8_t task;
	uint8_t group;
	uint8_t GroupMemberAddress;
	grouper = 0;
	for(task=1;task<=8;task++)
	{
		if(task != IndexTask)  // we are trying to find the OTHER members of the group
		{
			ie_eeprom_read_block(&group, &ee.EE_I2C_ButtonGroup[task-1],1);		
			if(group == IndexGroup)
			{
				ie_eeprom_read_block(&GroupMemberAddress, &ee.EE_I2C_ButtonAddress[task-1],1);		
				if(GroupMemberAddress==Address) grouper |= 1 << (task-1);
			}
		}
	}
}




void icRxTx()
{
	if(Transmit == I2CPreviousTransmit) return;
	I2CPreviousTransmit = Transmit;
	if(PA3AKE_AttendB > 0) icSetBPF();
	if(EnablePCA9555>0) icApplyRxTxDataword();
}


void icApplyRxTxDataword()
{
	uint8_t it = 0; // position in list of 8 IC2 tasks ( 1 to 8)
	uint8_t Opcode;
	uint8_t Group;
	uint8_t Address;
	uint16_t Dataword;
	for(it=1;it<=8;it++)
	{
		ie_eeprom_read_block(&Opcode, &ee.EE_I2C_ButtonOpcode[it-1],1);		
		if(Opcode <= 10) break;
		ie_eeprom_read_block(&Group, &ee.EE_I2C_ButtonGroup[it-1],1);		
		ie_eeprom_read_block(&Address, &ee.EE_I2C_ButtonAddress[it-1],1);		
		ie_eeprom_read_block(&Dataword, &ee.EE_I2C_ButtonDataword[it-1],2);		
		if( (((Opcode==11)||(Opcode==13)) && (Transmit==0)) 
		 || (((Opcode==12)||(Opcode==14)) && (Transmit==1)) )
		{
			icPCA9555ModifyBits('C', Address, Dataword);
		}
		else
		{
			icPCA9555ModifyBits('W', Address, Dataword);
		}
	}		
}



/*
 frmI2C.TaskCol_Address.Items from Hobcat
 
 inactive
Set bit(s) specified in data word
Clear bit(s) specified in data word
Toggle bit(s) specified in data word
Load data word 
Load upper byte
Load lower byte  
Group dataword step
Group upper byte step
Group lower byte step
Load data word at system start
Set bit(s) (Tx only:  clear on Rx)
Set bit(s) (Rx only:  clear on Tx)
Toggle bit(s)  (Tx only:  clear on Rx)
Toggle bit(s)  (Rx only:  clear on Tx)

*/


void icPCA9555Task(uint8_t I2Ctask)  // I2Ctask if the index (1 to 8) of the eight IC2 button tasks
{
	if(EnablePCA9555==0) return;
	uint8_t t;      // position in button task list 0 to 75+  (other groupo members)
	uint8_t  task;  // position in button task list 0 to 75+  
	uint8_t it = 0; // position in list of 8 IC2 tasks ( 1 to 8)
	uint8_t Opcode;
	uint8_t Group;
	uint8_t Address;
	uint8_t toggledon;
	uint16_t Dataword;
	uint16_t GroupMemberDataword;
	ie_eeprom_read_block(&Opcode, &ee.EE_I2C_ButtonOpcode[I2Ctask-1],1);		
	ie_eeprom_read_block(&Group, &ee.EE_I2C_ButtonGroup[I2Ctask-1],1);		
	ie_eeprom_read_block(&Address, &ee.EE_I2C_ButtonAddress[I2Ctask-1],1);		
	ie_eeprom_read_block(&Dataword, &ee.EE_I2C_ButtonDataword[I2Ctask-1],2);		
	task = I2Ctask + 64;   // task is now the position in the button task lists (0-75+)
	switch(Opcode)
	{
	  case  1:
		case 11:
		case 12:
		{
			if(Group!=0)
			{
				icFindGroupMembers(task,Group, Address);  // must have same address AND group no and not this task no
				for(it=1;it<=8;it++)   // t = task scan
				{
					t = it + 64;
					if(grouper & (1<<(it-1)))
					{
						ie_eeprom_read_block(&GroupMemberDataword, &ee.EE_I2C_ButtonDataword[it-1],2);	
						if(Opcode==1)
						icPCA9555ModifyBits('C', Address, GroupMemberDataword);
						else
						icModifyRxTxDatawordBits('C', GroupMemberDataword);		
						icApplyRxTxDataword();
						if(zt) etRThighlight(t,0);
						buRedrawLabel(t,0); // 0 = NOT Highlighted
					}
				}
			}
 			if(Opcode==1)
			icPCA9555ModifyBits('S', Address, Dataword);  // 'S' = Set:   | Dataword
			else
			{
				icModifyRxTxDatawordBits('S', Dataword);
				icApplyRxTxDataword();
			}			
			if(zt) etRThighlight(task,1);


			buRedrawLabel(task,1); // 1 = Highlighted
			break;
		}
	  case  2:
		{
 			icPCA9555ModifyBits('C', Address, Dataword);  // 'C' = Clear  & (~Dataword) 
			break;
	  }
		case  3: 	
		{
			toggledon = icPCA9555ModifyBits('T', Address, Dataword);  // 'T' = Toggle  ^ Dataword			
			if (zt) etRThighlight(task,toggledon);   // toggledon > highlighted
			buRedrawLabel(task,toggledon);
			break;
		}
		case 13:
		case 14:
		{
			toggledon = icModifyRxTxDatawordBits('T', Dataword);
			icApplyRxTxDataword();
			if (zt) etRThighlight(task,toggledon);   // toggledon > highlighted
			buRedrawLabel(task,toggledon);
			break;
	  }
		case  4: icPCA9555WriteBits('B', Address, Dataword); break;
	  case  5: icPCA9555WriteBits('U',Address, Dataword); break;
	  case  6: icPCA9555WriteBits('L', Address, Dataword); break;
	  case  7:
		case  8:
		case  9: 
		{
			if(Group==0) break;
			icFindGroupMembers(0,Group,Address); // Index = 0 so will find whole group
			if (grouper==0) break;  // otherwise search for next step goes into infinite loop
			if(GroupStepTask[Group-1]==0) GroupStepTask[Group-1] = task;  // start at this task
			do
			{
				GroupStepTask[Group-1] += 1;
				if(GroupStepTask[Group-1] > 8) GroupStepTask[Group-1] = 1;
				t = GroupStepTask[Group-1];
			}
			while((grouper & (1 << (t-1)))==0);
	 		ie_eeprom_read_block(&Dataword, &ee.EE_I2C_ButtonDataword[t-1],2);		
			// the label (2nd line used) of the task (only one of n tasks requires a button)
			icGetI2CGetButtonLabel(t,StrB,StepTaskLabel[Group-1]);  // StrB is a dummy here
			UseStepTaskLabel[Group-1] = 1;   
			buRedrawLabel(task, 0); // redraw the button pressed
			switch(Opcode)
			{
				case  7: icPCA9555WriteBits('B', Address, Dataword); break;
	  		case  8: icPCA9555WriteBits('U',Address, Dataword); break;
	  		case  9: icPCA9555WriteBits('L', Address, Dataword); break;
			}
			break;
		}
	  case 10: icPCA9555WriteBits('B', Address, Dataword); break;
	}
}




uint8_t icGetStepTaskGroup(uint8_t I2Ctask)
{
	uint8_t Group;
	ie_eeprom_read_block(&Group, &ee.EE_I2C_ButtonGroup[I2Ctask-1],1);		
	return Group;
}



void	icGetI2CGetButtonLabel(uint8_t I2Ctask, char* sa, char* sb)
{
	sb[0] = 0;
	ie_eeprom_read_block(sa, &ee.EE_I2C_ButtonLabels[I2Ctask-1][0],12);
	if(strlen(sa)>11){sa[11] = 0;};
	char* pc = strchr(sa,'-');
	if(pc>0)
	{
		strcpy(sb,pc+1);
		pc[0] = 0;
	}
}


void icGetI2CGetTaskLabel(uint8_t n,char* sa)
{
	ie_eeprom_read_block(sa, &ee.EE_I2C_ButtonLabels[n-1][0],12);
	if(strlen(sa)>11){sa[11] = 0;};
}



void icPA3AKE_AttenSwitch(char IncDec)
{
	if(IncDec=='+')
	{
		PA3AKE_AttendB += 2;
		if(PA3AKE_AttendB > 30) PA3AKE_AttendB = 0;
	}
	else
	{
		if(PA3AKE_AttendB < 2) PA3AKE_AttendB = 32; 
		PA3AKE_AttendB -= 2;
	}
	icSetBPF();
}

////////   Button label info ///////
char* icAttenLevelString()
{
	itoa(PA3AKE_AttendB,StrB,10);
	strcat(StrB,"dB");
	return StrB;
}


char* icRooferModeString()
{
	if(PA3AKE_Roofer=='S') strcpy(StrB,PMS(s_SSB)); else strcpy(StrB,PMS(s_CW));
  return StrB;
}

