/*
 *****    homebrew-radios.net   *****

 *****     Project TrxAVR       *****


 taDDS.c    program file
 
Support for AD9951 DDS sythesiser chip 
 
*/

#include "taGlobal.h"
#include <math.h>
#include "taConfig.h"
#include "taDDS.h"
#include "taDebug.h"
#include "taStarControl.h"
#include "taStarDSP.h"
#include "ta24LC512.h"
#include "taEncoder.h"
#include "taCharLCD.h"
#include "taFT245R.h"
#include "taGraphicsControl.h"
#include "taIntEeprom.h"
#include "taZ_DDS.h"
#include "taZ_StarControl.h"
#include "taDDScommands.h"
#include "taStarDSP.h"
#include "taZ_DDScommands.h"
#include "taEEPROM.h"		// for EEMEM definitions
#include "taEATFTmenus.h"


#define ddsFilterLow  24000000
#define ddsFilterHigh 44000000
#define port_ddsFilter PORTA
#define ddsFilter   2

#define port_ddsUD   PORTB
#define port_ddsSDIO PORTH
#define port_ddsCLK  PORTG
#define ddsUD     7
#define ddsSDIO   7
#define ddsLOclk  3
#define ddsCIOclk 4

struct DDSparams_type DDSparams;
struct Slot_type VfoSlot[2]; 
struct Slot_type RamSlot;

uint32_t fDDSclock;  //   DDS clock in Hz
uint32_t USBoffset;
uint32_t LSBoffset;
uint32_t CWoffset;
uint8_t CWoffsetEnable;
uint8_t DDSmultiplier;
uint32_t LOfreq;
uint32_t LOoffset;
uint8_t Split;
uint8_t PreviousSplit;
uint8_t VFO;
uint8_t IP3;
uint8_t PreviousVFO;
uint32_t Freq;
uint32_t TxFreq;  //used for XIT T tuning
uint32_t PriorTxFreq;
uint32_t StartRITTxFreq;
uint32_t RITfreq;
uint32_t PreviousFreq;
uint8_t Band;
uint8_t PreviousBand;
uint32_t VFOfreq[2];
uint8_t  VFOband[2];
uint8_t FreqChangeFlag;
uint8_t NeedShowFreq;
uint8_t UpdateDDS;
uint8_t LockDDS;
uint8_t HowToShowFreq;  // 'V' = vfo   'S' = stack
uint8_t VolatileSP;
uint8_t VolatileCount;
uint8_t VfoSlotNo;
uint8_t MemStackNo;
uint8_t MemSlotNo;
uint8_t SigGenMode;
uint8_t ClockCalMode;
uint8_t LowSideInjection;

 //Band lo switch points    160, 80, 60, 40, 30, 20,   17,  15,  12,  10
static const uint16_t BandLo[10] = {100,250,450,600,850,1200,1550,1950,2300,2650}; // x 10kHz


// F+M slot - 8 bytes - could do in 5 bytes - neater in eight

// ***************************************
//    VFO indexing:   A = 0  VFO B = 1
// ***************************************

static struct Slot_type VfoVolatileStack[23];  // RAM based slots  (not sticky)

/*		Moved to taEEPROM.h
struct Slot_type EEMEM EE_BandStack[3][10];  // EEPROM Sticky slots x 30   (60m slots unused)
struct Slot_type EEMEM EE_MemStack[4][6];   // EEPROM Sticky slots x 24
struct Slot_type EEMEM EE_VfoStickyStack[8];      // EEPROM Sticky slots  0 = PowerOn,   1-7 = VFO slots 1-7


//////////////////////////////////////
uint32_t EEMEM EE_fDDSclock = 174998221;  //   DDS clock in Hz
uint32_t EEMEM EE_USBoffset = 9012850;
uint32_t EEMEM EE_LSBoffset = 9010150;
*/

static uint8_t ButtonId;
static uint8_t ButtonData;
static uint8_t ButtonVfo;
static uint64_t ddsq;


void ddsSendSlotDataToPC();
void ddsReceiveFrequencyFromPC();
void ddsReceiveButtonFromPC();
void ddsHobcatBand();
void ddsHobcatMode();
void ddsHobcatFilter();
void ddsHobcatParamColour();
void ddsHobcatIP3();
void ddsHobcatXIT();
void ddsHobcatRIT();
void ddsHobcatSplit();
void ddsHobcatVfoSwap();
void ddsHobcatAtten();
void ddsHobcatDenoiser();
void ddsAD9951start();
void ddsSendBits(uint32_t *data, uint8_t nbits, char clk);


uint32_t ddsGetVfoFreq(uint8_t v) __attribute__((section(".highmem")));
uint32_t ddsGetVfoFreq(uint8_t v)
{
	return VfoSlot[v].stFreq;
}


void ddsSetVfoFreq(uint8_t v, uint32_t f) __attribute__((section(".highmem")));
void ddsSetVfoFreq(uint8_t v, uint32_t f)
{
	VfoSlot[v].stFreq = f;
}


// Sticky VFO stack
void ddsWriteOldestStickyVfoSlotNo(uint8_t n) __attribute__((section(".highmem")));
void ddsWriteOldestStickyVfoSlotNo(uint8_t n)
{
	xeWriteByte(xeGetOldestStickyVfoSlotNoAddr(), n);
}


uint8_t ddsReadOldestStickyVfoSlotNo() __attribute__((section(".highmem")));
uint8_t ddsReadOldestStickyVfoSlotNo()
{
	uint8_t n;

	xeReadByte(xeGetOldestStickyVfoSlotNoAddr(), &n);
	return n;
}


//////  Routines to update slot and save as BandLast
void ddsSaveModeToLast() __attribute__((section(".highmem")));
void ddsSaveModeToLast()
{
	VfoSlot[VFO].stMode = Mode;
	ddsSaveBandSlot(VFO, 'L', Band);
}


void ddsSaveFilterToLast() __attribute__((section(".highmem")));
void ddsSaveFilterToLast()
{
	VfoSlot[VFO].stFilterNarrow = SWfilter;
	ddsSaveBandSlot(VFO, 'L', Band);

}


void ddsSaveIP3ToLast() __attribute__((section(".highmem")));
void ddsSaveIP3ToLast()
{
	VfoSlot[VFO].stIP3 = IP3;
	ddsSaveBandSlot(VFO, 'L', Band);
}


void ddsSaveFreqToLast() __attribute__((section(".highmem")));
void ddsSaveFreqToLast()
{
	VfoSlot[VFO].stFreq = Freq;
	ddsSaveBandSlot(VFO, 'L', Band);
}


void ddsSaveParamColourToLast() __attribute__((section(".highmem")));
void ddsSaveParamColourToLast()
{
	VfoSlot[VFO].stParamColour = ParamColour;
	ddsSaveBandSlot(VFO, 'L', Band);
}


void ddsCopyVfo() __attribute__((section(".highmem")));
void ddsCopyVfo()
{
	if (VFO == 0) VfoSlot[1] = VfoSlot[0];
	else VfoSlot[0] = VfoSlot[1];
	VFO = 0;
}


void ddsClearClocks() __attribute__((section(".highmem")));
void ddsClearClocks()
{
	port_ddsCLK &= ~((1 << ddsLOclk) | (1 << ddsCIOclk));
}


void ddsInit() __attribute__((section(".highmem")));
void ddsInit()
{
	// clear clock lines early - before DDS hardware reset ends
	LowSideInjection = 0;
	VolatileSP =22;  // makes it start at slot0
	VolatileCount = 0;
	LockDDS = 0;
	TuningMode = 'N';
	MemStackNo = 3; //   60   - access to 60m
	MemSlotNo = 255;  // no slot
	Split = 0;
	Transmit = 0;
	PreviousBand = 255;     // force update
	PreviousTransmitF = 0;
	PreviousTransmitD = 0;
	PreviousSplit = 0;
	PreviousVFO = 255;
	NeedShowFreq = 1;
	FreqChangeFlag = 0;
	HowToShowFreq = 'V';  // VFO
//	ddsLoadDDSmultiplier();
	ie_eeprom_read_block(&fDDSclock, &ee.EE_fDDSclock, 4);
	ie_eeprom_read_block(&USBoffset, &ee.EE_USBoffset, 4);
	ie_eeprom_read_block(&LSBoffset, &ee.EE_LSBoffset, 4);
	ie_eeprom_read_block(&CWoffset, &ee.EE_CWoffset, 4);
	ie_eeprom_read_block(&CWoffsetEnable, &ee.EE_CWoffsetEnable, 1);
	ddsq =  0x8000000000000000 / fDDSclock;  // 2^63 / fDDSclock   
	ddsAD9951start();
	return;
}


void ddsAD9951start() __attribute__((section(".highmem")));
void ddsAD9951start()
{
	uint8_t multA;
	uint8_t multB;
	uint8_t bits012;
	uint32_t CFR2;
	uint32_t dataword;

	multA = HardwareSettings.hsDDSmultiplierA;
	multB = HardwareSettings.hsDDSmultiplierB;
	if ((multA ^ multB) != 0xFF) return;
	if ((multA < 4) || (multA > 20)) return;
	if (multA == 0) return;  // if no multiplier then all control words are defaults
	if (fDDSclock > 250000000) bits012 = 6; else bits012 = 2; // bit1  - charge pump 125uA
	CFR2 = (multA*8 + bits012);  
	dataword = 0x01000000;
	ddsSendBits(&dataword, 8, 'L');
	dataword = CFR2*0x100;   // shift 24 bytes into most significant
	ddsSendBits(&dataword, 24, 'L');
}



void ddsUpdateDDSclock(uint8_t save) __attribute__((section(".highmem")));
void ddsUpdateDDSclock(uint8_t save)
{
	if (save == 1) ie_eeprom_write_block(&fDDSclock, &ee.EE_fDDSclock, 4);
	ddsq =  0x8000000000000000 / fDDSclock;  // 2^63 / fDDSclock 
	ddsSetFreq();
}


void ddsUpdateLSBoffset(uint8_t save) __attribute__((section(".highmem")));
void ddsUpdateLSBoffset(uint8_t save)
{
	if (save == 1) ie_eeprom_write_block(&LSBoffset, &ee.EE_LSBoffset, 4);
	if ((Mode & 0x01) == 0) LOoffset=LSBoffset;
	ddsSetFreq();  
}  


void ddsUpdateUSBoffset(uint8_t save) __attribute__((section(".highmem")));
void ddsUpdateUSBoffset(uint8_t save)
{
	if (save == 1) ie_eeprom_write_block(&USBoffset, &ee.EE_USBoffset, 4);
	if ((Mode & 0x01) == 0x01) LOoffset=USBoffset;
	ddsSetFreq();
}  


void ddsUpdateCWoffset(uint8_t save) __attribute__((section(".highmem")));
void ddsUpdateCWoffset(uint8_t save)
{
	if (save == 1) {
		ie_eeprom_write_block(&CWoffset, &ee.EE_CWoffset, 4);
		ie_eeprom_write_block(&CWoffsetEnable, &ee.EE_CWoffsetEnable, 1);
	}
	if (((Mode & 0x02) == 0x02) && (CWoffsetEnable == 1)) LOoffset=CWoffset;
	ddsSetFreq();
}


// send the most significant nbits bytes of a 32 bit word
void ddsSendBits(uint32_t *data, uint8_t nbits, char clk) __attribute__((section(".highmem")));
void ddsSendBits(uint32_t *data, uint8_t nbits, char clk)
{
	int b;
	uint32_t mask;
	uint8_t clkmask;

	if (clk == 'L') clkmask = (1 << ddsLOclk); else clkmask = (1 << ddsCIOclk);
	for (b = 31; b >= (32-nbits); b--) {
		mask = 1;
		if ((*data & (mask << b)) > 0) port_ddsSDIO |= (1 << ddsSDIO);
		else port_ddsSDIO &= ~(1 << ddsSDIO);
		_delay_us(2);
		port_ddsCLK |= clkmask;
		_delay_us(1);  //1
		port_ddsCLK &= ~clkmask;
		_delay_us(2);//2
	}
	return;
}


void ddsSendFreq(uint32_t *hz, char clk) __attribute__((section(".highmem")));
void ddsSendFreq(uint32_t *hz, char clk)
{
	uint64_t q;
	uint32_t dataword;
	uint8_t cSREG;

	cSREG = SREG;
	cli();
	q = (*hz * ddsq) / 0x80000000;  // divide result by 2^31  
	// FTW address = 0x04. bit 7 clear for write
	dataword = 0x04000000;
	ddsSendBits(&dataword,8,clk);
	dataword = q;
	ddsSendBits(&dataword,32,clk);
	_delay_us(1);
	port_ddsUD |= (1 << ddsUD);
	_delay_us(1);
	port_ddsUD &= ~(1 << ddsUD);
	fdw = dataword;
	SREG = cSREG;
	return;
}


void ddsCheckRITshift() __attribute__((section(".highmem")));
void ddsCheckRITshift()
{
	if ((Transmit == 0) && (RIT == 1) && (Split == 0)) {
		if (labs(Freq-StartRITTxFreq) > 2500) cmCopyVfo(); // cancel RIT
	}
}


void ddsFreqChange() __attribute__((section(".highmem")));
void ddsFreqChange()
{
	VfoSlot[VFO].stFreq = Freq;
	Band = ddsFreqToBand(Freq);
	ddsSetFreq();  // will set UpdateDDS = 1 if fails
	NeedShowFreq = 1;
}


// Picastar uses BandLast on band change - even after power down
// except for power on band!!

void ddsCheckFreq() __attribute__((section(".highmem")));
void ddsCheckFreq()
{
	uint8_t TxVFO = VFO;

	/////////   BAND CHANGE  ////////////////////

	///////////   VFO A / B CHANGE /////////////////
	Band = ddsFreqToBand(Freq);
	if (Band != PreviousBand) {
		scSetBand(); // simply switch BPF and inform DSP
		             // but this change is on frequency only so no slot info
		PreviousBand = Band;
	}
	if ((Split != PreviousSplit) || (VFO != PreviousVFO) || // VFO does NOT change on split transmit (VFO is the A or B slot)
		(Band != PreviousBand) || (Transmit !=PreviousTransmitF)) {
		if ((Transmit != PreviousTransmitF) && (Split == 1)) {
			if (Transmit == 1) {
				if (VFO == 1) TxVFO = 0; else TxVFO = 1;
				ddsApplyFreq(TxVFO); // was ddsApplyVfo(TxVFO);  IJS 11.10.2009
				zddsDisplayGraphicsFreqs(TxVFO);
			} else { 
				ddsApplyFreq(VFO); // was ddsApplyVfo(VFO);  IJS 11.10.2009
				zddsDisplayGraphicsFreqs(VFO);
			}
		} else {
			UpdateDDS = 1;
		}
	} else {
		if (Transmit == 0) VfoSlot[VFO].stFreq = Freq;
	}
	PreviousVFO = VFO;
	PreviousSplit = Split;
	PreviousTransmitF = Transmit;
	if ((NeedShowFreq == 1) || (UpdateDDS == 1)) {  // encoder changes DDS directly and sets this
		zddsShowFreq();
		NeedShowFreq = 0;
	}
	if (UpdateDDS == 1) { // this comes from anywhere other than tuning encoder in 'N' mode
		ddsSetFreq();  // will clear UpdateDDS
	}
}


uint8_t ddsSetFreq() __attribute__((section(".highmem")));
uint8_t ddsSetFreq()
{
	uint32_t F;

	ddsSetOffset();
	if (TxXITtune == 1) F = TxFreq; else F = Freq;  // IJS 11.10.2009  changed Freq to F below
	if (LockDDS == 1) {UpdateDDS = 1; return 0;}
	LockDDS = 1;                // Avoid reentrancy and consequent DDS dropout
	if ((SigGenMode == 1) || (ClockCalMode == 1)) {
		LOfreq = Freq;
	} else {
		if (LowSideInjection) {
			if (F < LOoffset) LOfreq = LOoffset - F;
			else LOfreq = F - LOoffset;
		} else {
			LOfreq = F + LOoffset;
		}
	}
	if ((LOfreq > ddsFilterLow) && (LOfreq < ddsFilterHigh)) port_ddsFilter |= (1 << ddsFilter);
	else port_ddsFilter &= ~(1 << ddsFilter);
	ddsSendFreq(&LOfreq, 'L');
	UpdateDDS = 0;
	LockDDS = 0;
	return 1;
}


void ddsSetOffset() __attribute__((section(".highmem")));
void ddsSetOffset()
{
	if (LowSideInjection == 1) {
		DspMode = (Mode ^ 0x01);  // exclusive OR on bit zero
		if (((Mode & 0x02) == 0x02) && (CWoffsetEnable==1)) {
			LOoffset = CWoffset;
		} else {
 	  		if ((Mode & 0x01) == 0) LOoffset=USBoffset;  else LOoffset=LSBoffset;
		}
	} else {
 	 	DspMode = Mode;
		if (((Mode & 0x02) == 0x02) && (CWoffsetEnable==1)) {
			LOoffset = CWoffset;
		} else {
			 if ((Mode & 0x01) == 0) LOoffset=LSBoffset; else LOoffset=USBoffset;
		}
	}
}


//////////   SLOT routines ////////////////////////////


//// Routines to load to RamSlot for slot menu display /////////

void ddsLoadMemSlotToRamSlot(uint8_t stack, uint8_t slot) __attribute__((section(".highmem")));
void ddsLoadMemSlotToRamSlot(uint8_t stack, uint8_t slot)
{
	ie_eeprom_read_block(&RamSlot, &ee.EE_MemStack[stack][slot],8);
}


void ddsLoadStickyVfoSlotToRamSlot(uint8_t slot) __attribute__((section(".highmem")));
void ddsLoadStickyVfoSlotToRamSlot(uint8_t slot)
{
	ie_eeprom_read_block(&RamSlot, &ee.EE_VfoStickyStack[slot],8);
}


void ddsLoadVolatileVfoSlotToRamSlot(uint8_t slot) __attribute__((section(".highmem")));
void ddsLoadVolatileVfoSlotToRamSlot(uint8_t slot)
{
	RamSlot =  VfoVolatileStack[slot];
}


// LSC  L= BandLAST, S = BandSSB   C =Band CW 
void ddsLoadBandSlotToRamSlot(char LSC, uint8_t band) __attribute__((section(".highmem")));
void ddsLoadBandSlotToRamSlot(char LSC, uint8_t band)
{
  uint8_t a = 0;
	switch(LSC)
	{
	  case 'L': a = 0; break;
		case 'S': a = 1; break;
		case 'C': a = 2; break;
	}
	ie_eeprom_read_block(&RamSlot, &ee.EE_BandStack[a][band],8);	  
}	  


////   slot <> VFO routines ///////////////////////

void ddsLoadPowerOnSlot(uint8_t vfo) __attribute__((section(".highmem")));
void ddsLoadPowerOnSlot(uint8_t vfo)
{
	uint8_t v = vfo;

	if (v == 2) v = 0;
	ie_eeprom_read_block(&VfoSlot[v], &ee.EE_VfoStickyStack[0], 8);
	if (vfo == 2) VfoSlot[1] = VfoSlot[0];
}


// vfo  0=VFOA  1=VFOB  2=both A&B
// LSC  L= BandLAST, S = BandSSB   C =Band CW 
void ddsLoadBandSlot(uint8_t vfo, char LSC, uint8_t band) __attribute__((section(".highmem")));
void ddsLoadBandSlot(uint8_t vfo, char LSC, uint8_t band)
{
  uint8_t a = 0;
	uint8_t v = vfo;
	switch(LSC)
	{
	  case 'L': a = 0; break;
		case 'S': a = 1; break;
		case 'C': a = 2; break;
	}
	if(v==2){v=0;};
	ie_eeprom_read_block(&VfoSlot[v], &ee.EE_BandStack[a][band],8);
	if(vfo==2){VfoSlot[1] = VfoSlot[0];};	  
}	  


// stacks are 7,8,9 and 0   slots are 1 to 6  eg: 84 is stack 8 slot 4
void ddsLoadMemSlot(uint8_t vfo, uint8_t stack, uint8_t slot) __attribute__((section(".highmem")));
void ddsLoadMemSlot(uint8_t vfo, uint8_t stack, uint8_t slot)
{
	uint8_t v = vfo;
	if(v==2){v=0;};
	ie_eeprom_read_block(&VfoSlot[v], &ee.EE_MemStack[stack][slot],8);
	if(vfo==2){VfoSlot[1] = VfoSlot[0];};	  
}	  



// VFO slots 1-7 are  poistions 1 - 7 in EEMEM (VFO slot 0 is actually the power up slot)
void ddsLoadStickyVfoSlot(uint8_t vfo, uint8_t slot) __attribute__((section(".highmem")));
void ddsLoadStickyVfoSlot(uint8_t vfo, uint8_t slot)
{
	uint8_t v = vfo;
	if(v==2){v=0;};
	ie_eeprom_read_block(&VfoSlot[vfo], &ee.EE_VfoStickyStack[slot],8);
	if(vfo==2){VfoSlot[1] = VfoSlot[0];};	  
}	  


// VFO slots 0-23 are RAM based
void ddsLoadVolatileVfoSlot(uint8_t vfo, uint8_t slot) __attribute__((section(".highmem")));
void ddsLoadVolatileVfoSlot(uint8_t vfo, uint8_t slot)
{
	uint8_t v = vfo;
	if(v==2){v=0;};
	VfoSlot[v] = VfoVolatileStack[slot];
	if(vfo==2){VfoSlot[1] = VfoSlot[0];};	  
}	  



void ddsSavePowerOnSlot(uint8_t vfo) __attribute__((section(".highmem")));
void ddsSavePowerOnSlot(uint8_t vfo)
{
  ie_eeprom_write_block(&VfoSlot[vfo], &ee.EE_VfoStickyStack[0], 8);
}

// vfo  0=SlotA  1=SlotB
// LSC  L= BandLAST, S = BandSSB   C =Band CW 
void ddsSaveBandSlot(uint8_t vfo, char LSC, uint8_t band) __attribute__((section(".highmem")));
void ddsSaveBandSlot(uint8_t vfo, char LSC, uint8_t band)
{
  uint8_t a = 0;
	switch(LSC)
	{
	  case 'L': a = 0; break;
		case 'S': a = 1; break;
		case 'C': a = 2; break;
	}
	ie_eeprom_write_block(&VfoSlot[vfo], &ee.EE_BandStack[a][band],8);	  
}


// stacks are 7,8,9 and 0   slots are 1 to 6  eg: 84 is stack 8 slot 4
void ddsSaveMemSlot(uint8_t vfo, uint8_t stack, uint8_t slot) __attribute__((section(".highmem")));
void ddsSaveMemSlot(uint8_t vfo, uint8_t stack, uint8_t slot)
{
	ie_eeprom_write_block(&VfoSlot[vfo], &ee.EE_MemStack[stack][slot],8);
}


// VFO slots 1-7 are  positions 1 - 7 in EEMEM (VFO slot 0 is actually the power up slot)
void ddsSaveStickyVfoSlot(uint8_t vfo, uint8_t slot) __attribute__((section(".highmem")));
void ddsSaveStickyVfoSlot(uint8_t vfo, uint8_t slot)
{
	ie_eeprom_write_block(&VfoSlot[vfo], &ee.EE_VfoStickyStack[slot],8);
}


// VFO slots 0-22 are RAM based
void ddsSaveVolatileVfoSlot(uint8_t vfo, uint8_t slot) __attribute__((section(".highmem")));
void ddsSaveVolatileVfoSlot(uint8_t vfo, uint8_t slot)
{
  VfoVolatileStack[slot] = VfoSlot[VFO];
}

////////////////////////////////////////////////////////

void ddsApplyVfo(uint8_t v) __attribute__((section(".highmem")));
void ddsApplyVfo(uint8_t v)
{
	uint8_t bl, SaveMode;

	Freq = VfoSlot[v].stFreq;
	Mode = VfoSlot[v].stMode;
	ParamColour = VfoSlot[v].stParamColour;
	IP3 = VfoSlot[v].stIP3;
	SWfilter = VfoSlot[v].stFilterNarrow;
	if (Mode < 2) SSB_SWfilter = SWfilter;
	else CW_SWfilter = 1; // SWfilter;  // IJS Star6a9 chnaged to 1 - depth   no context
	ddsSetFreq();
	Band = ddsFreqToBand(Freq);
	bl = dspSendBulkLoad(1);
	if (Band != PreviousBand) {
		scSetBand();
		PreviousBand = Band;
		_delay_ms(200);
	}
	if (Mode != PreviousMode) {
		if (PreviousMode == 255) {
			dspTidySwitches();	// set all switches to 0
			if (Mode > 1) {
				SaveMode = Mode;
				Mode &= 0xfd;
				scChangeMode();
				dspSendCommand(6, 0);
				if (bl) dspSendBulkLoad(0);
				bl = dspSendBulkLoad(1);
				Mode = SaveMode;
				ddsSetFreq();
			}
		}
		scChangeMode();
		PreviousMode = Mode;
	} else {
		dspSetParameterColour();
		dspSendCommand(2, DspMode);
	}
	scSetIP3(IP3);
	dspSendCommand(6, SWfilter);
	if (Mode < 2) dspSetSSBfilterCentre(63-SWfilter);
	if (bl) dspSendBulkLoad(0);
	dspLoadTxDrive(); // IJSmod 25082012
	zscShowSwitches();
	zscShowFilter();
	zscDisplayIP3NF();
	zscDisplayParamColour(ParamColour);
}


// new routine  IJS 11.10.2009
void ddsApplyFreq(uint8_t v) __attribute__((section(".highmem")));
void ddsApplyFreq(uint8_t v)
{
	Freq = VfoSlot[v].stFreq;
	ddsSetFreq();
	Band = ddsFreqToBand(Freq);
	if (Band != PreviousBand) {
		scSetBand();
		PreviousBand = Band;
		_delay_ms(200);
	}
}


uint8_t ddsFreqToBand(uint32_t freq) __attribute__((section(".highmem")));
uint8_t ddsFreqToBand(uint32_t freq)
{
  uint8_t b = 9;
  uint16_t tenk = freq/10000;
  while((BandLo[b] > tenk)&&(b>0)){b-=1;};
  return b;
}


//   SAVE and LOAD slots to from PC ///////////////////////////////////////////////

void ddsStackLoadRequest(char stk) __attribute__((section(".highmem")));
void ddsStackLoadRequest(char stk)
{
  uint8_t block[96];
	switch(stk)
	{
		case 'L': 
		  {
		    if(usbReceiveBlock((uint8_t*)(&block),80,500)==1)
				{ 
          ie_eeprom_write_block(block, &ee.EE_BandStack[0][0], 80);
				}
				break;
      }
		case 'S': 
		  {
		    if(usbReceiveBlock((uint8_t*)(&block),80,500)==1)
				{ 
          ie_eeprom_write_block(block, &ee.EE_BandStack[1][0], 80);
				}
				break;
      }
		case 'C': 
		  {
		    if(usbReceiveBlock((uint8_t*)(&block),80,500)==1)
				{ 
          ie_eeprom_write_block(block, &ee.EE_BandStack[2][0], 80);
				}
				break;
      }
		case '7': 
		  {
		    if(usbReceiveBlock((uint8_t*)(&block),48,500)==1)
				{ 
          ie_eeprom_write_block(block, &ee.EE_MemStack[0][0], 48);
				}
				break;
      }
		case '8': 
		  {
		    if(usbReceiveBlock((uint8_t*)(&block),48,500)==1)
				{ 
          ie_eeprom_write_block(block, &ee.EE_MemStack[1][0], 48);
				}
				break;
      }
		case '9': 
		  {
		    if(usbReceiveBlock((uint8_t*)(&block),48,500)==1)
				{ 
          ie_eeprom_write_block(block, &ee.EE_MemStack[2][0], 48);
				}
				break;
      }
		case '0': 
		  {
		    if(usbReceiveBlock((uint8_t*)(&block),48,500)==1)
				{ 
          ie_eeprom_write_block(block, &ee.EE_MemStack[3][0], 48);
				}
				break;
      }
		case 'V': 
		  {
		    if(usbReceiveBlock((uint8_t*)(&block),64,500)==1)
				{ 
          ie_eeprom_write_block(block, &ee.EE_VfoStickyStack[0], 64);
				}
				break;
      }

  }
}

void ddsStackSaveRequest(char stk) __attribute__((section(".highmem")));
void ddsStackSaveRequest(char stk)
{
  uint8_t block[96];
	switch(stk)
	{
		case 'L': 
		  {
        ie_eeprom_read_block(block, &ee.EE_BandStack[0][0], 80);
		    usbSendBlock((uint8_t*)(&block),80,500);
				break;
      }

		case 'S': 
		  {
        ie_eeprom_read_block(block, &ee.EE_BandStack[1][0], 80);
		    usbSendBlock((uint8_t*)(&block),80,500);
				break;
      }
		case 'C': 
		  {
        ie_eeprom_read_block(block, &ee.EE_BandStack[2][0], 80);
		    usbSendBlock((uint8_t*)(&block),80,500);
				break;
      }
		case '7': 
		  {
        ie_eeprom_read_block(block, &ee.EE_MemStack[0][0], 48);
		    usbSendBlock((uint8_t*)(&block),48,500);
				break;
      }
		case '8': 
		  {
        ie_eeprom_read_block(block, &ee.EE_MemStack[1][0], 48);
		    usbSendBlock((uint8_t*)(&block),48,500);
				break;
      }
		case '9': 
		  {
        ie_eeprom_read_block(block, &ee.EE_MemStack[2][0], 48);
		    usbSendBlock((uint8_t*)(&block),48,500);
				break;
      }
		case '0': 
		  {
        ie_eeprom_read_block(block, &ee.EE_MemStack[3][0], 48);
		    usbSendBlock((uint8_t*)(&block),48,500);
				break;
      }
		case 'V': 
		  {
        ie_eeprom_read_block(block, &ee.EE_VfoStickyStack[0], 64);
		    usbSendBlock((uint8_t*)(&block),64,500);
				break;
      }

  }

}


////////////////////////////////////////////////////////////////////////////////
/////////////DDS CAT ////////////////////////////////

void ddsUsbRequest(char cmd) __attribute__((section(".highmem")));
void ddsUsbRequest(char cmd)
{
  switch(cmd)
	{
	  case 'S': ddsSendSlotDataToPC(); break;
	  case 'F': ddsReceiveFrequencyFromPC(); break;
		case 'B': ddsReceiveButtonFromPC(); break;
  }
}


void ddsSendSlotDataToPC() __attribute__((section(".highmem")));
void ddsSendSlotDataToPC()
{
	uint8_t buf[28];
	memcpy(&buf[0],&VfoSlot[0],8);
	memcpy(&buf[8],&VfoSlot[1],8);
  buf[16] = VFO;
	buf[17] = Transmit;
	buf[18] = Mode;
	buf[19] = RIT;
	buf[20] = XIT;
	buf[21] = Split;
	buf[22] = ddsFreqToBand(VfoSlot[0].stFreq);
	buf[23] = ddsFreqToBand(VfoSlot[1].stFreq);
	buf[24] = SmeterdB;
	buf[25] = RFatten;
	buf[26] = SWdenoise;
	buf[27]=0;
	usbSendBlock(buf,28,1000); 
}


void ddsReceiveFrequencyFromPC() __attribute__((section(".highmem")));
void ddsReceiveFrequencyFromPC()
{
  uint32_t F;
	uint8_t buf[6];
  if(usbReceiveBlock(buf,5,5000)==0){return;};
  uint8_t v = buf[4];  // 5th byte is VFO no
  memcpy(&F,&buf[0],4);  // first four bytes are freq
	ddsExtSetFreq(v,F);
	if(v==VFO){ClearSlotno = 1;};
}

void ddsExtSetFreq(uint8_t v, uint32_t Fext) __attribute__((section(".highmem")));
void ddsExtSetFreq(uint8_t v, uint32_t Fext)
{
	VfoSlot[v].stFreq=Fext;
	if(v==VFO)
	{
		Freq =  Fext;
		NeedShowFreq = 1;
		if(ddsSetFreq()==0)  // will set UpdateDDS = 0 if successful
		{
			UpdateDDS = 1; // if locked out then set this flag to update in main loop
  	};
  }
	else
	{ 
	  zddsShowInactiveVfoFreq();
  };

}

void ddsReceiveButtonFromPC() __attribute__((section(".highmem")));
void ddsReceiveButtonFromPC()
{
  uint8_t buf[4];
  if(usbReceiveBlock(buf,3,5000)==0){return;};
  ButtonId = buf[0];
	ButtonVfo = buf[1];
  ButtonData = buf[2];
  switch(ButtonId)
	{
		case 'Q': ddsHobcatBand(); break;   // band change
    
		case 'M': ddsHobcatMode(); break;

		case 'F': ddsHobcatFilter(); break;

		case 'P': ddsHobcatParamColour(); break;

		case 'I': ddsHobcatIP3(); break; 		

		case 'X': ddsHobcatXIT(); break;
		
		case 'R': ddsHobcatRIT(); break;

		case 'S': ddsHobcatSplit(); break;

		case 'A': ddsHobcatVfoA(); break;
		
		case 'B': ddsHobcatVfoB(); break;

		case 'W': ddsHobcatVfoSwap(); break;

		case '>': ddsHobcatAtoB(); break;

		case '<': ddsHobcatBtoA(); break;
	
		case 'T': ddsHobcatAtten(); break;
	
		case 'N': ddsHobcatDenoiser(); break;	
	}
	switch(ButtonId)
	{
		case 'I':
		case 'P':
		case 'T': if (zt) {emeDeleteInfo(); emeDrawDateTime();}; 
	}
  ddsSendSlotDataToPC();
}


void ddsHobcatBand() __attribute__((section(".highmem")));
void ddsHobcatBand()
{
  if(VFO==ButtonVfo)
	{
		if(ButtonData==1){cmBandUp();}else{cmBandDown();};
	}
	else
	{
  	uint8_t B = ddsFreqToBand(VfoSlot[ButtonVfo].stFreq);
		if(ButtonData==1)    // band up
		{
			B+=1;
			if(B==2){B=3;}; // skip 60m
	  	if(Band > 9){Band = 0;};
		}
		else
		{
		  if(B>0){B-=1;}else{B=9;};
			if(B==2){B=1;};
		}
		ddsLoadBandSlot(ButtonVfo, 'L', B); 
		ClearSlotno = 1;
		zddsShowInactiveVfoFreq();
	}
}


void ddsHobcatFilter() __attribute__((section(".highmem")));
void ddsHobcatFilter()   // Toggle copmmand received
{
  if(VFO==ButtonVfo)
	{
	 	scWideNarrow(); 
	}
	else
	{
		VfoSlot[ButtonVfo].stFilterNarrow ^= 0x01; 
	}
}


void ddsHobcatMode() __attribute__((section(".highmem")));
void ddsHobcatMode()   // Toggle copmmand received
{
  if(VFO==ButtonVfo)
	{
	 	Mode = ButtonData;
		scChangeMode(); 
		ddsSaveModeToLast(); 
		zscShowFilter();
		zscShowSwitches();
	}
	else
	{
		VfoSlot[ButtonVfo].stMode = ButtonData; 
	}
}




void ddsHobcatParamColour() __attribute__((section(".highmem")));
void ddsHobcatParamColour()   /// toggle command receieved
{

	if(VFO==ButtonVfo)
	{
	 	switch(ButtonData)
		{
		  case (1): ParamColour = pgreen; break; 
		  case (2): ParamColour = pyellow; break; 
		  case (3): ParamColour = pred; break; 
    }
  	ddsSaveParamColourToLast(); 
	  dspSetParameterColour();
		zscDisplayIP3NF();
	  zscShowFilter();
	}
	else
	{    
		VfoSlot[ButtonVfo].stParamColour = ButtonData; 
	}
}


void ddsHobcatIP3() __attribute__((section(".highmem")));
void ddsHobcatIP3()
{
  if(VFO==ButtonVfo)
	{
	  if(IP3==1){IP3=0;} else {IP3=1;};
		scSetIP3(IP3);						
		zscDisplayIP3NF();
		ddsSaveIP3ToLast(); 
	}
	else
	{    
		VfoSlot[ButtonVfo].stIP3 ^= 0x01; 
	}

}


void ddsHobcatXIT() __attribute__((section(".highmem")));
void ddsHobcatXIT()
{
	if(XIT==1)
	{cmCancelRITXIT();}
	else
	{cmXIT();};
}


void ddsHobcatRIT() __attribute__((section(".highmem")));
void ddsHobcatRIT()
{
	if(RIT==1){cmCancelRITXIT(); return;};
	if(RIT==0)
	{
		if(XIT==1){cmCancelRITXIT();};
		cmRIT();
	}
}


void ddsHobcatSplit() __attribute__((section(".highmem")));
void ddsHobcatSplit()
{
  if(XRIT()==0){ return;};
	if(Split==1){Split=0;}else{Split=1;};
	//Split ^= 0x01;
	ddsApplyVfo(VFO);
	UpdateDDS = 1;
	zddsShowBothVfoFreqs();   
	zcmShowSplitMode();      
}


void ddsHobcatVfoA() __attribute__((section(".highmem")));
void ddsHobcatVfoA()
{
  VFO = 0;
	ddsApplyVfo(VFO); 
	UpdateDDS = 1;
	zddsShowBothVfoFreqs();
	ClearSlotno = 1;
}


void ddsHobcatVfoB() __attribute__((section(".highmem")));
void ddsHobcatVfoB()
{
  VFO = 1;
	ddsApplyVfo(VFO); 
	UpdateDDS = 1;
	zddsShowBothVfoFreqs();
	ClearSlotno = 1;
}


void ddsHobcatVfoSwap() __attribute__((section(".highmem")));
void ddsHobcatVfoSwap()
{
	struct Slot_type TempSlot;
	TempSlot = VfoSlot[0];
	VfoSlot[0] = VfoSlot[1];
	VfoSlot[1] = TempSlot;
	ddsApplyVfo(VFO); 
	UpdateDDS = 1;
	zddsShowBothVfoFreqs();
	ClearSlotno = 1;
}


void ddsHobcatAtoB() __attribute__((section(".highmem")));
void ddsHobcatAtoB()
{
	VfoSlot[1] = VfoSlot[0];
	ddsApplyVfo(VFO); 
	UpdateDDS = 1;
	zddsShowBothVfoFreqs();
	ClearSlotno = 1;
}


void ddsHobcatBtoA() __attribute__((section(".highmem")));
void ddsHobcatBtoA()
{
	VfoSlot[0] = VfoSlot[1];
	ddsApplyVfo(VFO); 
	UpdateDDS = 1;
	zddsShowBothVfoFreqs();
	ClearSlotno = 1;
}



void ddsHobcatAtten() __attribute__((section(".highmem")));
void ddsHobcatAtten()
{
	cmAtten();
}


void ddsHobcatDenoiser() __attribute__((section(".highmem")));
void ddsHobcatDenoiser()
{
	scDenoise();
}
