/*
 *****    homebrew-radios.net   *****

 *****     Project TxrAVR       *****


 taCharLCD.c    program file
 
Character display driver (40x2  OR 20x4 chars)
XY coordinate input for 40x2  and optionally converted to 20x4    
NT3881D driver chip
 
*/

#include "taGlobal.h"
#include "taCharLCD.h"
#include "taFT245R.h"

/////////////////  control pins ///////////

// pin mapping for standard char LCD (only) control
//  R/W = PG0 
//  E   = PG1 
//  RS  = PD7

// pin mapping for dual display char LCD control  
// R/W is to be physically tied to Gnd (Write only)
// E    = PL1
// RS   = PL2


//   data port
#define ddrChData  DDRC
#define portChDATA PORTC

// RS, RW and E port char LCD only
#define portChRS PORTD
#define ChRS 7
#define portChRW PORTG
#define ChRW 0
#define portChE  PORTG
#define ChE  1

// RS and E port dual display (graphics on standard port)
#define portChRSdual PORTL
#define ChRSdual 2
#define portChEdual  PORTL
#define ChEdual  1


/////////////////////////////////////////////////

#define databits 1  // 8 bit parallel interface
#define dualline 1  // two lines
#define font 0      // 5x8 dots

#define cursormove 1  // increment cursor position after write
#define shift 0       // display shift off
#define rightshift 1  // shift right if shift is on
#define displayon 1   // display on
#define cursoron 0    // cursor off
#define blinking 0    // cursor blinking - if it were on


static const char CGRAM[64] PROGMEM = {
	0x0E,0x11,0x0E,0x00,0x0A,0x15,0x00,0x1F,  // character 0 = 130 (watts - vertical)
	0x1B,0x1B,0x1B,0x1B,0x1B,0x00,0x00,0x00,  // character 1 = 5x5 pixels
	0x18,0x18,0x18,0x18,0x18,0x00,0x00,0x00,  // character 2 = left half pixels
	0x0E,0x11,0x0E,0x00,0x00,0x1F,0x00,0x00,  // character 3 = 10 (+10dB)
	0x0E,0x11,0x0E,0x00,0x0D,0x15,0x13,0x00,  // character 4 = 20 (+20dB)
	0x0E,0x11,0x0E,0x00,0x0A,0x15,0x15,0x00,  // character 5 = 30 (+30dB)
	0x0E,0x11,0x0E,0x02,0x1F,0x0A,0x06,0x00,  // character 6 = 40 (+40dB)
	0x16,0x15,0x0E,0x00,0x0D,0x15,0x13,0x00,  // character 7 = 26  (watts - verical)
};


void cdInitCharLCD()
{  
	uint8_t i;

	if (!zc) return;
	_delay_ms(20);
	cdCommand(0x20 | (databits << 4)); // switch to 8 bit mode before using bits 0-3 !! 
	_delay_ms(5);
	cdCommand(0x20 | (databits << 4)); // then repeat
	cdCommand(0x20 | (databits << 4)); // then repeat
	cdCommand(0x20 | (databits << 4) | (dualline << 3) | (font << 2));
	cdCommand(0x08); // Display off
	cdCommand(0x04 | (cursormove << 1) | (shift << 0));
	cdCommand(0x10 | (shift << 3) | (rightshift << 2));

	for (i = 0; i < 0x40; i++)    // Smeter chars   ascii codes 1 to 5
	{
		cdCommand(0x40 + i);   // CGRAM character 1  - right and left side vertical bar
		cdPutByte(1, pgm_read_byte(&CGRAM[i]));
	}

	cdCommand(0x08 | (displayon << 2) | (cursoron << 1) | (blinking << 0));
	cdClear();
	cdHome();
	return;
}

uint8_t cdDisplayString(char *s)
{
	if (!zc) return 0;
	int nc = strlen(s);
	for (int i = 0; i < nc; i++)
	{
		cdPutByte(1, s[i]);
	}   	
	return nc;
}

void cdHome()
{
	if (!zc) return;
	cdCommand(2);
	_delay_ms(2);  // needs 1.64mS minimum delay
	return;
}


void cdClear()
{
	if (!zc) return;
	cdCommand(1);
	_delay_ms(2);  // needs 1.64mS minimum delay
	return;
}

void cdClearLeft()
{
	if (zc)
	{
		cdSetPos(1, 0);
		cdDisplayString(pmdSpaces(20));
		cdSetPos(2, 0);
		cdDisplayString(pmdSpaces(20));
	}
}


void cdShowCursor(uint8_t con)
{
  if (!zc) return;
  if (con == 0) cdCommand(0x0C); else cdCommand(0x0E);
}


// character write position  lines 1 and 2, position 0 - 39
void cdSetPos(unsigned char line, unsigned char pos)
{
	if (!zc) return;
	if (TwentyByFour == 1)
	{
		if (pos < 20) line += 2; else pos -= 20; // translate 40x2 to 20x4
	}
	switch(line)
	{
		case 1: cdCommand(0x80 + pos); break;
		case 2: cdCommand(0xC0 + pos); break;
		case 3: cdCommand(0x94 + pos); break;
		case 4: cdCommand(0xD4 + pos); break;
	}
}





void cdCommand(unsigned char b)
{
	if (!zc) return;
	cdPutByte(0, b);
	return;
}



// set RS line to 0 (instruction) or 1 (data) and write byte b
void cdPutByte(unsigned char rs, unsigned char b)
{
	if (!zc) return;
	usbINTenable(3,0); // disable Rx interrupt INT3
	if (DualDisplay == 0) {
		portChRW &= ~(1 << ChRW);  // clear RW bit = set for writing
		if (rs == 0) portChRS &= ~(1 << ChRS);  // RS to logical 0
		else portChRS |= (1 << ChRS);  // RS to logical 1
	}
	else {
		if (rs == 0) portChRSdual &= ~(1 << ChRSdual);  // RS to logical 0
		else portChRSdual |= (1 << ChRSdual);  // RS to logical 1
	}
  	ddrChData = 0xFF;            // set port to output
	portChDATA = b;    // set up the data lines          
	if (DualDisplay == 0)
	{
		portChE |= (1 << ChE);	// E to logical 1   ( pulse the E line)
		//	3 * 3 * 62.5ns delay
		__asm__ volatile (
			"ldi r24,3 \r\n"
			"1: dec r24 \r\n"
			"brne 1b \r\n"
			: : : "r24"
		);
		portChE &= ~(1 << ChE);	// E to logical 0
	}  
	else
	{
		portChEdual |= (1 << ChEdual);	// E to logical 1   ( pulse the E line)
		//	3 * 3 * 62.5ns delay
		__asm__ volatile (
			"ldi r24,3 \r\n"
			"1: dec r24 \r\n"
			"brne 1b \r\n"
			: : : "r24"
		);
		portChEdual &= ~(1 << ChEdual);	// E to logical 0
	}
	if (DualDisplay == 0) portChRS |= (1 << ChRS);  // RS to logical 1  - other wise not left highZ
	else portChRSdual |= (1 << ChRSdual);  // RS to logical 1  - other wise not left highZ
	if (RxIntEnabled == 1) usbINTenable(3,1); // enable Rx interrupt INT3
	_delay_us(40);		// minimum cycle time = 40uS	
	return;
}


/*

Bit-wise operations in C

We need to set or clear single or multiple bits of a port 
where the bit(s) are specified bit bit numbers 0 - 7
There is no function do do this - (but easy to write one)

(1 << 4) starts with the number 1 and shifts it left 4 positions 
 and so is equal to 0x10 ie: bit 4 set.

 So if b is a byte
 Then b = (1 << 4); sets b = 0x10
 and b = b | (1 << 4);    ORs bit 4 of b with 1 ie sets bit 4 and preserves the state of the rest
 This can be written as b |= (1 << 4);

 To clear bit 4 we AND b with 0x10 with bits inverted
 ie:  b &= ~(1 << 4);     // ~ inverts the bits
 To clear bits 4 and 5
      b &= ~((1 <<4) | (1 <<5));  ie AND b with a mask which has bits 4 and 5 cleared

In cdPutByte we have

	portChE |= (1 << ChE);     // E to logical 1   ( pulse the E line)
	cdDelay(2);
	portChE &= ~(1 << ChE);    // E to logical 0

ChE is simply a constant predefined for convenience  ( easy to change without altering code)
so ~(1 << ChE) is a constant and so is evaluated by the compiler, and not at run time.
ChE = 1 (bit 1), so (1 << ChE) = 0x02  ( 00000010), so the inverted ~(1 << ChE) = 0xFD  ( 1111 1101)
So what is compiled is PORTG = PORTG & 0xFE

*/
