//
//  PROGRAM NAME: tf_font
//
//  Converts *.FXT files to a format suitable for loading to TftA
//
//  Invocation: tf_font <font list file> <TftA font file>
//
//  <font list file>    input file containing the list of font.fxt files and their index
//  <TftA font file>    output file for loading to TftA via Hobcat
//
//  Input file format:
//
//  FONTS: <font filepath>  - where <font filepath> provides a path to the *.fxt files
//  nnn fontname.fxt  - where nnn is the font index (1 - 128) and fontname.fxt is the matching font file
//                    - one line for every font to be included
//                    - each line a maximum of 259 characters
//  ; comment         - comment lines start with a ';' in column 1
//

#include "tf_font.h"

//
// Global variables
//
static int pfile = 0;

//
// Forward declarations of procedures
//
static long process_font(ULONG font_index, UCHAR *fontfile, ULONG total_bytes, FILE *streamout, UCHAR *fileout);
static void print_msg(char *msgi, unsigned long line);

//
// Main routine
//
int main(USHORT argc, PCHAR argv[])
{
	ULONG			i, count, font_index, font_size, font_ptr[128];
	ULONG			rc = 0, line_count = 0, total_bytes = 0, font_cnt = 0, zero = 0;
	FILE			*streamin, *streamout;
	fpos_t			file_start, file_end;
	UCHAR			*buf_ptr, command[6], buffer[260], msg[501];
	UCHAR			filein[256], fileout[256], fontfile[256], font_path[256], font[512];

	pfile = open("tf_font.LOG", O_APPEND | O_CREAT | O_BINARY | O_RDWR, S_IREAD | S_IWRITE);
	if (pfile == -1) {
		pfile = 0;
	}

	print_msg("***** ENTRY tf_font (last modified on "__TIMESTAMP__ " and compiled on " __DATE__ " "  __TIME__ ")", __LINE__);

	if ( (argc == 3) && (strlen(argv[1]) > 0) && (strlen(argv[2]) > 0) ) {
		strcpy(filein, argv[1]);
		strcpy(fileout, argv[2]);
		if (NULL == (streamin = fopen(filein, "rt"))) {
			sprintf(msg, "Error \"%s\" occurred executing fopen for %s", strerror(errno), filein);
			print_msg(msg, __LINE__);
			print_msg("***** EXIT tf_font", __LINE__);
			if (pfile != 0) close(pfile);
			return 8;
		}
		if (NULL == (streamout = fopen(fileout, "wb+"))) {
			sprintf(msg, "Error \"%s\" occurred executing fopen for %s", strerror(errno), fileout);
			print_msg(msg, __LINE__);
			fclose(streamin);
			print_msg("***** EXIT tf_font", __LINE__);
			if (pfile != 0) close(pfile);
			return 8;
		}
		if (fgetpos(streamout, &file_start)) {
			sprintf(msg, "Error \"%s\" occurred executing fgetpos (1) for %s", strerror(errno), fileout);
			print_msg(msg, __LINE__);
			fclose(streamin);
			fclose(streamout);
			print_msg("***** EXIT tf_font", __LINE__);
			if (pfile != 0) close(pfile);
			return 8;
		}
		
		memcpy(command, "\x1b\x45\x00\x00\x00\x00", 6);
		count = fwrite(command, 1, 6, streamout);
		if ((ferror(streamout)) || (count != 6)) {
			sprintf(msg, "Error \"%s\" occurred executing fwrite for %s", strerror(errno), fileout);
			print_msg(msg, __LINE__);
			fclose(streamin);
			fclose(streamout);
			print_msg("***** EXIT tf_font", __LINE__);
			if (pfile != 0) close(pfile);
			return 8;
		}

		for (i = 0; i < 128; i++) {
			font_ptr[i] = 0;
			count = fwrite(&font_ptr[i], 1, sizeof(ULONG), streamout);
			total_bytes += sizeof(ULONG);
			if ((ferror(streamout)) || (count != sizeof(ULONG))) {
				sprintf(msg, "Error \"%s\" occurred executing fwrite for %s", strerror(errno), fileout);
				print_msg(msg, __LINE__);
				fclose(streamin);
				fclose(streamout);
				print_msg("***** EXIT tf_font", __LINE__);
				if (pfile != 0) close(pfile);
				return 8;
			}
		}

		while ( !feof(streamin) ) {
			buf_ptr = fgets(buffer, sizeof(buffer), streamin);
			if (buf_ptr != NULL) {
				line_count++;
				if (buffer[0] == ';') {
					continue;
				}
				if (!strncmp(buffer, "FONTS: ", 7)) {
					buf_ptr = strchr(buffer, ':');
					do {buf_ptr++;} while (isspace(*buf_ptr));
					strcpy(font_path, buf_ptr);
					buf_ptr = font_path + strlen(font_path);
					do {buf_ptr--;} while (isspace(*buf_ptr));
					*(buf_ptr + 1) = 0;
					strcat(font_path, "\\");
					continue;
				}
				count = sscanf(buffer, "%lu %s", &font_index, fontfile);
				if ((count != 2) || ((font_index < 1) || (font_index > 128)) || (strlen(fontfile) == 0)) {
					sprintf(msg, "Invalid line (%d): \"%s\"", line_count, buffer);
					print_msg(msg, __LINE__);
				} else {
					strcpy(font, font_path);
					strcat(font, fontfile);
					sprintf(msg, "Processing font file: \"%s\"  index: %d", font, font_index);
					print_msg(msg, __LINE__);
					font_size = process_font(font_index, font, total_bytes, streamout, fileout);
					if (font_size != 0) {
						if (font_ptr[font_index - 1] != 0) {
							sprintf(msg, "Warning: font index %d has been reused - original font will be overwritten", font_index);
							print_msg(msg, __LINE__);
						}
						font_ptr[font_index - 1] = total_bytes;
						if (font_size & 0x00000001) {
							font_size++;
							count = fwrite(&zero, 1, 1, streamout);
							if ((ferror(streamout)) || (count != 1)) {
								sprintf(msg, "Error \"%s\" occurred executing fwrite for %s", strerror(errno), fileout);
								print_msg(msg, __LINE__);
								fclose(streamin);
								fclose(streamout);
								print_msg("***** EXIT tf_font", __LINE__);
								if (pfile != 0) close(pfile);
								return 8;
							}
						}
						total_bytes += font_size;
						font_cnt++;
					} else {
						rc = 8;
						break;
					}
				}
			}
			if ( ferror(streamin) ) {
				sprintf(msg, "Error \"%s\" occurred executing fgets for %s", strerror(errno), filein);
				print_msg(msg, __LINE__);
				rc = 8;
				break;
			}
		}
		
		if (fgetpos(streamout, &file_end)) {
			sprintf(msg, "Error \"%s\" occurred executing fgetpos (2) for %s", strerror(errno), fileout);
			print_msg(msg, __LINE__);
			fclose(streamin);
			fclose(streamout);
			print_msg("***** EXIT tf_font", __LINE__);
			if (pfile != 0) close(pfile);
			return 8;
		}

		if (fsetpos(streamout, &file_start)) {
			sprintf(msg, "Error \"%s\" occurred executing fsetpos (1) for %s", strerror(errno), fileout);
			print_msg(msg, __LINE__);
			fclose(streamin);
			fclose(streamout);
			print_msg("***** EXIT tf_font", __LINE__);
			if (pfile != 0) close(pfile);
			return 8;
		}

		memcpy(&command[2], &total_bytes, 4);
		count = fwrite(command, 1, 6, streamout);
		if ((ferror(streamout)) || (count != 6)) {
			sprintf(msg, "Error \"%s\" occurred executing fwrite for %s", strerror(errno), fileout);
			print_msg(msg, __LINE__);
			fclose(streamin);
			fclose(streamout);
			print_msg("***** EXIT tf_font", __LINE__);
			if (pfile != 0) close(pfile);
			return 8;
		}

		for (i = 0; i < 128; i++) {
			count = fwrite(&font_ptr[i], 1, sizeof(ULONG), streamout);
			if ((ferror(streamout)) || (count != sizeof(ULONG))) {
				sprintf(msg, "Error \"%s\" occurred executing fwrite for %s", strerror(errno), fileout);
				print_msg(msg, __LINE__);
				fclose(streamin);
				fclose(streamout);
				print_msg("***** EXIT tf_font", __LINE__);
				if (pfile != 0) close(pfile);
				return 8;
			}
		}

		if (fsetpos(streamout, &file_end)) {
			sprintf(msg, "Error \"%s\" occurred executing fsetpos (2) for %s", strerror(errno), fileout);
			print_msg(msg, __LINE__);
			fclose(streamin);
			fclose(streamout);
			print_msg("***** EXIT tf_font", __LINE__);
			if (pfile != 0) close(pfile);
			return 8;
		}
		
		fclose(streamin);
		fclose(streamout);
		sprintf(msg, "Fonts processed: %d, bytes written: %d", font_cnt, total_bytes);
		print_msg(msg, __LINE__);
	} else {
		print_msg("Usage: tf_font \"font list file\" \"TftA font file\"", __LINE__);
		print_msg("       The quotes are required around the filenames if", __LINE__);
		print_msg("       you are using long filenames", __LINE__);
		print_msg("***** EXIT tf_font", __LINE__);
		if (pfile != 0) close(pfile);
		return (4);
	}

	print_msg("***** EXIT tf_font", __LINE__);

	if (pfile != 0) close(pfile);

	return rc;

}

//
// Process font files
//
static long process_font(ULONG font_index, UCHAR *fontfile, ULONG total_bytes, FILE *streamout, UCHAR *fileout)
{
	FILE			*fontin;
	ULONG			font_bytes = 0, line_count = 0, count = 0, first, temp, row_bytes;
	ULONG			char_offset, char_bytes, char_index, mode, height_count, width, i;
	UCHAR			first_char, last_char, max_width, height, font_type;
	UCHAR			*buf_ptr, buffer[65536], line[501], lines[8][501], msg[501];

	for (i = 0; i < sizeof(buffer); i++) {
		buffer[i] = 0;
	}
	
	if (NULL == (fontin = fopen(fontfile, "rt"))) {
		sprintf(msg, "Error \"%s\" occurred executing fopen for %s", strerror(errno), fontfile);
		print_msg(msg, __LINE__);
		return 0;
	}

	while ( !feof(fontin) ) {
		buf_ptr = fgets(line, sizeof(line), fontin);
		if ( ferror(fontin) ) {
			sprintf(msg, "Error \"%s\" occurred executing fgets for %s", strerror(errno), fontfile);
			print_msg(msg, __LINE__);
			fclose(fontin);
			return 0;
		}
		if (buf_ptr != NULL) {
			if (line[0] != ';') {
				break;
			}
			if (line_count == 8) {
				sprintf(msg, "Invalid font file format (1) - File: \"%s\", too many header lines", fontfile);
				print_msg(msg, __LINE__);
				fclose(fontin);
				return 0;
			}
			strcpy(lines[line_count], line);
			line_count++;
		} else {
			sprintf(msg, "Invalid font file format (2) - File: \"%s\"", fontfile);
			print_msg(msg, __LINE__);
			fclose(fontin);
			return 0;
		}
	}
	
	if (line_count < 5) {
		sprintf(msg, "Invalid font file format (3) - File: \"%s\", insufficient header lines", fontfile);
		print_msg(msg, __LINE__);
		fclose(fontin);
		return 0;
	}
	
	buf_ptr = strrchr(lines[line_count - 5], ':');
	if (buf_ptr == NULL) {
		sprintf(msg, "Invalid font file format (4a) - File: \"%s\"", fontfile);
		print_msg(msg, __LINE__);
		fclose(fontin);
		return 0;
	}
	count = sscanf(buf_ptr, ": %lu ", &temp);
	if ((count != 1) || (temp > 255)) {
		sprintf(msg, "Invalid font file format (4b) - File: \"%s\", first char value invalid: %lu", fontfile, temp);
		print_msg(msg, __LINE__);
		fclose(fontin);
		return 0;
	}
	first_char = (UCHAR)temp;
	
	buf_ptr = strrchr(lines[line_count - 4], ':');
	if (buf_ptr == NULL) {
		sprintf(msg, "Invalid font file format (5a) - File: \"%s\"", fontfile);
		print_msg(msg, __LINE__);
		fclose(fontin);
		return 0;
	}
	count = sscanf(buf_ptr, ": %lu ", &temp);
	if ((count != 1) || (temp > 255)) {
		sprintf(msg, "Invalid font file format (5b) - File: \"%s\", last char value invalid: %lu", fontfile, temp);
		print_msg(msg, __LINE__);
		fclose(fontin);
		return 0;
	}
	last_char = (UCHAR)temp;
	
	buf_ptr = strrchr(lines[line_count - 3], ':');
	if (buf_ptr == NULL) {
		sprintf(msg, "Invalid font file format (6a) - File: \"%s\"", fontfile);
		print_msg(msg, __LINE__);
		fclose(fontin);
		return 0;
	}
	count = sscanf(buf_ptr, ": %s ", &buffer);
	if ((count != 1) || !((strcmp(buffer, "monospaced")) || (strcmp(buffer, "proportional")))) {
		sprintf(msg, "Invalid font file format (6b) - File: \"%s\", font type invalid: %s", fontfile, buffer);
		print_msg(msg, __LINE__);
		fclose(fontin);
		return 0;
	}
	font_type = (strcmp(buffer, "monospaced")) ? 1 : 0;
	
	buf_ptr = strrchr(lines[line_count - 2], ':');
	if (buf_ptr == NULL) {
		sprintf(msg, "Invalid font file format (7a) - File: \"%s\"", fontfile);
		print_msg(msg, __LINE__);
		fclose(fontin);
		return 0;
	}
	count = sscanf(buf_ptr, ": %lu ", &temp);
	if ((count != 1) || (temp == 0) ||(temp > 255)) {
		sprintf(msg, "Invalid font file format (7b) - File: \"%s\", width value invalid: %lu", fontfile, temp);
		print_msg(msg, __LINE__);
		fclose(fontin);
		return 0;
	}
	if (temp > 127) {
		sprintf(msg, "Invalid font file format (7c) - File: \"%s\", width value > 127: %lu", fontfile, temp);
		print_msg(msg, __LINE__);
		fclose(fontin);
		return 0;
	}
	max_width = (UCHAR)temp;
	
	buf_ptr = strrchr(lines[line_count - 1], ':');
	if (buf_ptr == NULL) {
		sprintf(msg, "Invalid font file format (8a) - File: \"%s\"", fontfile);
		print_msg(msg, __LINE__);
		fclose(fontin);
		return 0;
	}
	count = sscanf(buf_ptr, ": %lu ", &temp);
	if ((count != 1) || (temp == 0) || (temp > 255)) {
		sprintf(msg, "Invalid font file format (8b) - File: \"%s\", height value invalid: %lu", fontfile, temp);
		print_msg(msg, __LINE__);
		fclose(fontin);
		return 0;
	}
	if (temp > 100) {
		sprintf(msg, "Invalid font file format (8c) - File: \"%s\", height value > 100: %lu", fontfile, temp);
		print_msg(msg, __LINE__);
		fclose(fontin);
		return 0;
	}
	height = (UCHAR)temp;
	
	if (last_char < first_char) {
		sprintf(msg, "Invalid font file format (9) - File: \"%s\", last char (%u) < first char(%u)",
			fontfile, last_char, first_char);
		print_msg(msg, __LINE__);
		fclose(fontin);
		return 0;
	}
	
	buffer[0] = height;
	buffer[1] = first_char;
	buffer[2] = last_char;
	buffer[3] = max_width | (font_type << 7);
	char_offset = ((last_char - first_char + 1) * 2) + 4;
	font_bytes = char_offset;
	char_index = first_char;
	char_bytes = 0;
	first = 1;
	mode = 0;
	
	while ( !feof(fontin) ) {
		if (!first) {
			buf_ptr = fgets(line, sizeof(line), fontin);
		} else first = 0;
		if ( ferror(fontin) ) {
			sprintf(msg, "Error \"%s\" occurred executing fgets for %s", strerror(errno), fontfile);
			print_msg(msg, __LINE__);
			fclose(fontin);
			return 0;
		}
		if (feof(fontin)) {
			continue;
		}
		if (buf_ptr != NULL) {
			line_count++;
			if (mode == 0) {
				if (line[0] == '\n') {
					continue;
				}
				count = sscanf(buf_ptr, "%lu ", &temp);
				if ((count != 1) || (temp > last_char) || (temp < char_index)) {
					sprintf(msg, "Invalid font file format (11a) - File: \"%s\", line %lu",
						fontfile, line_count);
					print_msg(msg, __LINE__);
					fclose(fontin);
					return 0;
				}
				if (temp != char_index) {
					sprintf(msg, "Skipping font characters - File: \"%s\", line %lu, char index %lu through %lu",
						fontfile, line_count, char_index, (temp - 1));
					print_msg(msg, __LINE__);
					char_index = temp;
				}
				mode = 1;
				height_count = 0;
				memcpy(&buffer[((char_index - first_char) * 2) + 4], &char_offset, 2);
				continue;
			} else {
				if (strlen(line) <= 1) {
					sprintf(msg, "Invalid font file format (11b) - File: \"%s\", line %lu, invalid font pattern: \"%s\"",
						fontfile, line_count, line);
					print_msg(msg, __LINE__);
					fclose(fontin);
					return 0;
				}
				if (mode == 1) {
					if ((char_offset + 1) >= sizeof(buffer)) {
						sprintf(msg, "Invalid font file format (11c) - File: \"%s\", line %lu, font too large",
							fontfile, line_count);
						print_msg(msg, __LINE__);
						fclose(fontin);
						return 0;
					}
					mode = 2;
					width = strlen(line) - 1;
					if (width > max_width) {
						sprintf(msg, "Invalid font file format (11d) - File: \"%s\", line %lu, font pattern width greater than header max width",
							fontfile, line_count);
						print_msg(msg, __LINE__);
						fclose(fontin);
						return 0;
					}
					row_bytes = ((width - 1) / 8) + 1;
					buffer[char_offset] = (UCHAR)width;
					buffer[char_offset + 1] = (UCHAR)row_bytes;
					char_offset += 2;
				}
				if ((strlen(line) - 1) != width) {
					sprintf(msg, "Invalid font file format (11e) - File: \"%s\", line %lu, font pattern width inconsistent",
						fontfile, line_count);
					print_msg(msg, __LINE__);
					fclose(fontin);
					return 0;
				}
				temp = 0;
				for (i = 0; i < width; i++) {
					if ((line[i] != '.') && (line[i] != '#')) {
						sprintf(msg, "Invalid font file format (11f) - File: \"%s\", line %lu, invalid font pattern chars: \"%s\"",
							fontfile, line_count, line);
						print_msg(msg, __LINE__);
						fclose(fontin);
						return 0;
					}
					if (line[i] == '#') {
						temp |= 1 << (i % 8);
					}
					if ((((i + 1) % 8) == 0) && ((i + 1) != width)) {
						if (char_offset >= sizeof(buffer)) {
							sprintf(msg, "Invalid font file format (11g) - File: \"%s\", line %lu, font too large",
								fontfile, line_count);
							print_msg(msg, __LINE__);
							fclose(fontin);
							return 0;
						}
						buffer[char_offset] = (UCHAR)temp;
						char_offset++;
						temp = 0;
					}
				}
				if (char_offset >= sizeof(buffer)) {
					sprintf(msg, "Invalid font file format (11h) - File: \"%s\", line %lu, font too large",
						fontfile, line_count);
					print_msg(msg, __LINE__);
					fclose(fontin);
					return 0;
				}
				buffer[char_offset] = (UCHAR)temp;
				char_offset++;
				height_count++;
				if (height_count == height) {
					mode = 0;
					char_index++;
				}
			}
		} else {
			sprintf(msg, "Invalid font file format (11i) - File: \"%s\"", fontfile);
			print_msg(msg, __LINE__);
			fclose(fontin);
			return 0;
		}
	}
	
//	font_bytes = sprintf(buffer, "Font: %s, First char: %d, Last char: %d, Max width: %d, Height: %d, Type: %s\r\n",
//		fontfile, first_char, last_char, max_width, height, font_type ? "proportional" : "monospaced");

	if (char_offset == font_bytes) {
		sprintf(msg, "Invalid font file format (12) - File: \"%s\", no font chars supplied", fontfile);
		print_msg(msg, __LINE__);
		fclose(fontin);
		return 0;
	}

	if ((char_index - 1) != last_char) {
		sprintf(msg, "Invalid font file format (13) - File: \"%s\", not all font chars supplied", fontfile);
		print_msg(msg, __LINE__);
		fclose(fontin);
		return 0;
	}

	font_bytes = char_offset;
	count = fwrite(buffer, 1, font_bytes, streamout);
	if ((ferror(streamout)) || (count != font_bytes)) {
		sprintf(msg, "Error \"%s\" occurred executing fwrite for %s", strerror(errno), fileout);
		print_msg(msg, __LINE__);
		font_bytes = 0;
	}

	fclose(fontin);
	return font_bytes;

}


//
// Error Message Display Routine
//
static void   print_msg(char * msgi, unsigned long line)
{
	long        count;
	char        print_buf[4096];
	SYSTEMTIME  systemtime;

	GetLocalTime(&systemtime);
	printf("%s\r\n", msgi);
	if (pfile != 0) {
		count = sprintf(print_buf, "(PID %lu TID %lu) %3.3s %2u %3.3s %4u %02u:%02u:%02u.%03u tf_font %s at line %lu in file %s.\r\n",
    	GetCurrentProcessId(), GetCurrentThreadId(),
      DAYS[systemtime.wDayOfWeek],
      systemtime.wDay,
      MONTHS[systemtime.wMonth],
      systemtime.wYear,
      systemtime.wHour,
      systemtime.wMinute,
      systemtime.wSecond,
      systemtime.wMilliseconds,
      msgi, line, __FILE__);
		write(pfile, print_buf, count);
	}

	return;

}
