/*
 *****    homebrew-radios.net   *****

 *****     Project Encoders8       *****

 en_Encoders.c  Provide interrupt driven encoders8 functionality

*/

#include "encoders8.h"
#include "en_Encoders.h"
#include "en_RS232.h"
#include <string.h>
#include "avr_compiler.h"

/* Encoder counters */
static int EncoderCount[8];
static uint8_t EncoderFlag;

static void enTimerInit();
static bool enSendCount(uint8_t nint, int8_t count);


// Encoder has phase and quadrature logic outputs
// Phase signal interrupts on falling edge
// Then check corresponding quadrature pin
// If quad is low - increment Encodercount
// If quad is high - decrement Encodercount
void enEncodersInit()
{
	uint8_t i;
	
	for (i = 0; i < 8; i++) EncoderCount[i] = 0;  //reset counters
	EncoderFlag = 0;

	enTimerInit();
	
	EIMSK = 0x00;  // Disable external interrupts 0 - 7
	
	DDRD = 0x00;   // Set port D pins 0 - 7 to input
	DDRE &= 0x0f;  // Set port E pins 4 - 7 to input
	DDRF &= 0xf0;  // Set port F pins 0 - 3 to input

	PORTD = 0xff;  // Enable pullups for port D pins 0 - 7
	PORTE |= 0xf0; // Enable pullups for port E pins 4 - 7
	PORTF |= 0x0f; // Enable pullups for port F pins 0 - 3

	EICRA = 0xaa;  // Enable interrupt on falling edge for external interrupts 0 - 3
	EICRB = 0xaa;  // Enable interrupt on falling edge for external interrupts 4 - 7
	EIMSK = 0xff;  // Enable external interrupts 0 - 7
	EIFR = 0xff;   // Clear any external interrupt flags

	return;
}


void enTimerInit()
{
	// Stop and clear counter
	TCCR1A = 0x02; // Fast PWM mode with ICR as TOP value
	TCCR1B = 0x18; // "
	TCCR1C = 0x00;
	TCNT1 = 0;
	
	// Set period/TOP value - 1ms with a DIV1 clock source
	ICR1 = 16000;

	// Enable overflow interrupt
	TIMSK1 = 0x01;

	return;
}


bool enSendCount(uint8_t nint, int8_t delta)
{
	uint8_t i;
	bool bRC = false;
	char sa[6], sb[4];
	
	sa[0] = '$';
	sa[1] = 0x30 + nint;
	if (delta < 0) sa[2] = '+'; else sa[2] = '-';
	delta = abs(delta);
	itoa(delta, sb, 10);
	if (delta > 9) {sa[3] = sb[0]; sa[4] = sb[1];}
	else {sa[3] = '0'; sa[4] = sb[0];}
	sa[5] = 13;
	cli();
	if (USART_TXBuffer_Space() >= 6) {
		for (i = 0; i < 6; i++) {USART_TX[USART_TX_Head++] = sa[i];}
		UCSR0B |= 0x20;  // enable DRE interrupts
		bRC = true;
	}
	sei();
	return bRC;
}


ISR(INT0_vect) {
	if (EncoderFlag & (uint8_t)0x01) return;
	if (EncoderFlag) EncoderFlag |= (uint8_t)0x01;
	else {
		EncoderFlag |= (uint8_t)0x01;
		TCNT1 = 0; // Clear Counter
		TCCR1B |= 0x01; // Start Counter
	}
	sei();	
	return;
}

ISR(INT1_vect) {
	if (EncoderFlag & (uint8_t)0x02) return;
	if (EncoderFlag) EncoderFlag |= (uint8_t)0x02;
	else {
		EncoderFlag |= (uint8_t)0x02;
		TCNT1 = 0; // Clear Counter
		TCCR1B |= 0x01; // Start Counter
	}
	sei();	
	return;
}

ISR(INT2_vect) {
	if (EncoderFlag & (uint8_t)0x04) return;
	if (EncoderFlag) EncoderFlag |= (uint8_t)0x04;
	else {
		EncoderFlag |= (uint8_t)0x04;
		TCNT1 = 0; // Clear Counter
		TCCR1B |= 0x01; // Start Counter
	}
	sei();	
	return;
}

ISR(INT3_vect) {
	if (EncoderFlag & (uint8_t)0x08) return;
	if (EncoderFlag) EncoderFlag |= (uint8_t)0x08;
	else {
		EncoderFlag |= (uint8_t)0x08;
		TCNT1 = 0; // Clear Counter
		TCCR1B |= 0x01; // Start Counter
	}
	sei();	
	return;
}

ISR(INT4_vect) {
	if (EncoderFlag & (uint8_t)0x10) return;
	if (EncoderFlag) EncoderFlag |= (uint8_t)0x10;
	else {
		EncoderFlag |= (uint8_t)0x10;
		TCNT1 = 0; // Clear Counter
		TCCR1B |= 0x01; // Start Counter
	}
	sei();	
	return;
}

ISR(INT5_vect) {
	if (EncoderFlag & (uint8_t)0x20) return;
	if (EncoderFlag) EncoderFlag |= (uint8_t)0x20;
	else {
		EncoderFlag |= (uint8_t)0x20;
		TCNT1 = 0; // Clear Counter
		TCCR1B |= 0x01; // Start Counter
	}
	sei();	
	return;
}

ISR(INT6_vect) {
	if (EncoderFlag & (uint8_t)0x40) return;
	if (EncoderFlag) EncoderFlag |= (uint8_t)0x40;
	else {
		EncoderFlag |= (uint8_t)0x40;
		TCNT1 = 0; // Clear Counter
		TCCR1B |= 0x01; // Start Counter
	}
	sei();	
	return;
}

ISR(INT7_vect) {
	if (EncoderFlag & (uint8_t)0x80) return;
	if (EncoderFlag) EncoderFlag |= (uint8_t)0x80;
	else {
		EncoderFlag |= (uint8_t)0x80;
		TCNT1 = 0; // Clear Counter
		TCCR1B |= 0x01; // Start Counter
	}
	sei();	
	return;
}

ISR(TIMER1_OVF_vect)
{
	// Stop and clear counter
	TCCR1B &= 0xf8;
	TCNT1 = 0; // Clear Counter
	
	if (EncoderFlag & (uint8_t)0x01) {
		EncoderFlag &= (uint8_t)~0x01;
		if (!(PIND & (uint8_t)0x01)) {
			if (PIND & (uint8_t)0x10) {
				if (EncoderCount[0] > -99) EncoderCount[0] -= 1;
			} else {
				if (EncoderCount[0] < 99) EncoderCount[0] += 1;
			}
			if (enSendCount(0, EncoderCount[0])) EncoderCount[0] = 0;
		}
		if (EncoderFlag) TCCR1B |= 0x01; // Start Counter
		sei();	
		return;
	}

	if (EncoderFlag & (uint8_t)0x02) {
		EncoderFlag &= (uint8_t)~0x02;
		if (!(PIND & (uint8_t)0x02)) {
			if (PIND & (uint8_t)0x20) {
				if (EncoderCount[1] > -99) EncoderCount[1] -= 1;
			} else {
				if (EncoderCount[1] < 99) EncoderCount[1] += 1;
			}
			if (enSendCount(1, EncoderCount[1])) EncoderCount[1] = 0;
		}
		if (EncoderFlag) TCCR1B |= 0x01; // Start Counter
		sei();	
		return;
	}

	if (EncoderFlag & (uint8_t)0x04) {
		EncoderFlag &= (uint8_t)~0x04;
		if (!(PIND & (uint8_t)0x04)) {
			if (PIND & (uint8_t)0x40) {
				if (EncoderCount[2] > -99) EncoderCount[2] -= 1;
			} else {
				if (EncoderCount[2] < 99) EncoderCount[2] += 1;
			}
			if (enSendCount(2, EncoderCount[2])) EncoderCount[2] = 0;
		}
		if (EncoderFlag) TCCR1B |= 0x01; // Start Counter
		sei();	
		return;
	}

	if (EncoderFlag & (uint8_t)0x08) {
		EncoderFlag &= (uint8_t)~0x08;
		if (!(PIND & (uint8_t)0x08)) {
			if (PIND & (uint8_t)0x80) {
				if (EncoderCount[3] > -99) EncoderCount[3] -= 1;
			} else {
				if (EncoderCount[3] < 99) EncoderCount[3] += 1;
			}
			if (enSendCount(3, EncoderCount[3])) EncoderCount[3] = 0;
		}
		if (EncoderFlag) TCCR1B |= 0x01; // Start Counter
		sei();	
		return;
	}

	if (EncoderFlag & (uint8_t)0x10) {
		EncoderFlag &= (uint8_t)~0x10;
		if (!(PINE & (uint8_t)0x10)) {
			if (PINF & (uint8_t)0x08) {
				if (EncoderCount[4] > -99) EncoderCount[4] -= 1;
			} else {
				if (EncoderCount[4] < 99) EncoderCount[4] += 1;
			}
			if (enSendCount(4, EncoderCount[4])) EncoderCount[4] = 0;
		}
		if (EncoderFlag) TCCR1B |= 0x01; // Start Counter
		sei();	
		return;
	}

	if (EncoderFlag & (uint8_t)0x20) {
		EncoderFlag &= (uint8_t)~0x20;
		if (!(PINE & (uint8_t)0x20)) {
			if (PINF & (uint8_t)0x04) {
				if (EncoderCount[5] > -99) EncoderCount[5] -= 1;
			} else {
				if (EncoderCount[5] < 99) EncoderCount[5] += 1;
			}
			if (enSendCount(5, EncoderCount[5])) EncoderCount[5] = 0;
		}
		if (EncoderFlag) TCCR1B |= 0x01; // Start Counter
		sei();	
		return;
	}

	if (EncoderFlag & (uint8_t)0x40) {
		EncoderFlag &= (uint8_t)~0x40;
		if (!(PINE & (uint8_t)0x40)) {
			if (PINF & (uint8_t)0x02) {
				if (EncoderCount[6] > -99) EncoderCount[6] -= 1;
			} else {
				if (EncoderCount[6] < 99) EncoderCount[6] += 1;
			}
			if (enSendCount(6, EncoderCount[6])) EncoderCount[6] = 0;
		}
		if (EncoderFlag) TCCR1B |= 0x01; // Start Counter
		sei();	
		return;
	}

	if (EncoderFlag & (uint8_t)0x80) {
		EncoderFlag &= (uint8_t)~0x80;
		if (!(PINE & (uint8_t)0x80)) {
			if (PINF & (uint8_t)0x01) {
				if (EncoderCount[7] > -99) EncoderCount[7] -= 1;
			} else {
				if (EncoderCount[7] < 99) EncoderCount[7] += 1;
			}
			if (enSendCount(7, EncoderCount[7])) EncoderCount[7] = 0;
		}
	}
	
	sei();	
	return;

}
